/*
 *	(j)cal.c - print calendar
 *
 *		Usage: (j)cal [-#jemt] [-|+] [[month] year]
 */

static char *rcsid = "$Id: cal.c 1.10 94/07/07 23:54:52 Sugi Exp Locker: Sugi $";

#ifndef __MSDOS__
#define stricmp strcasecmp
#define strnicmp strncasecmp
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "common.h"
#include "progname.h"

#define PROGNAME		"cal"

#define LANG			"LANG"
#define JAPANESE		"japanese"
#ifdef MULTILINGUAL
#define FRENCH			"french"
#define SPANISH			"spanish"
#define GERMAN			"german"
#endif
#define CALTABENV		"CALTAB"
#ifdef __MSDOS__
#define DEFCALTAB		"/etc/caltab"
#else
# define DEFCALTAB		"/usr/local/etc/caltab"
#endif
#define COMMENT			'#'
#define YEAREFMT		"%d"
#define YEARJFMT		"%dǯ"

#define NOVAL			(-1)

#define C1WIDTH			3
#define C2WIDTH			4
#define MARGIN			4
#define DEFCOL			2
#define MAXCOL			3
#define MAXWIDTH		(C2WIDTH * 7 * MAXCOL + (MARGIN * (MAXCOL - 1)))

#define ROWMONTH		0
#define ROWHEADER		1
#define ROW1WEEK		2
#define CALBUFROW		(ROW1WEEK + MONTHWEEK)
#define CALBUFSIZE		(C2WIDTH * 7 + 1)

#define ENGMODE			0
#ifdef MULTILINGUAL
#define FREMODE			1
#define SPAMODE			2
#define GERMODE			3
#define JAPMODE			4
#define MAXLANG			5
#else
#define JAPMODE			1
#define MAXLANG			2
#endif
#define MONTHWEEK		6
#define JURIUSGAP		12

#define READBUFSIZE		128

#define adjdw(dw)		((dw) > 6 ? 0 : (dw))
#define setholiday(m,d)	(holtab[(m) - 1] |= (1L << ((d) - 1)))
#define isholiday(m,d)	(holtab[(m) - 1] & (1L << ((d) - 1)))

static char *progname;

static BOOL t_opt = FALSE;
static int lang = ENGMODE;
static int linelen = C2WIDTH * 7;
static int abbr;
static int headertype;
static char *monthname[MAXLANG][12] = {
	{
		"January",	"February",	"March",
		"April",	"May",		"June",
		"July",   	"August",	"September",
		"October",	"November",	"December"
	},
#ifdef MULTILINGUAL
	{
		"janvier",	"fevrier",	"mars",
		"avril",	"mai",		"juin",
		"juillet",	"aout",		"septembre",
		"octobre",	"novembre",	"decembre"
	},
	{
		"enero",	"febrero",	"marzo",
		"abril",	"mayo",		"junio",
		"julio",	"agosto",	"septiembre",
		"octubre",	"novienbre","diciembre"
	},
	{
		"Januar",	"Februar",	"Marz",
		"April",	"Mai",		"Juni",
		"Juli",		"August",	"September",
		"Oktober",	"November",	"Dezember"
	},
#endif
	{
#ifdef NOALIAS
		"1",		"2",		"3",
		"4",		"5",		"6",
		"7",		"8",		"9",
		"10",		"11",		"12"
#else
		"ӷ 1",	"ǡ 2",	" 3",
		" 4",	" 5",	"̵ 6",
		"ʸ 7",	"շ 8",	"Ĺ 9",
#ifdef IZUMO
		"߷ 10",
#else
		"̵ 10",
#endif
					" 11"," 12"
#endif
	}
};
static char *header[MAXLANG][2] = {
	{"Sun Mon Tue Wed Thu Fri Sat ", "Su Mo Tu We Th Fr Sa"},
#ifdef MULTILINGUAL
	{"dim lun mar mer jeu ven sam ", "di lu ma me je ve sa"},
	{"dom lun mar mie jue vie sab ", "do lu ma mi ju vi sa"},
	{"Son Mon Die Mit Don Fre Sam ", "So Mo Di Mi Do Fr Sa"},
#endif
	{"              ", "      "}
};
static long holtab[12] = {0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L};
static char calbuf[MAXCOL][CALBUFROW][CALBUFSIZE];

char obuf[BUFSIZ];

void cal(int, int, int, int);
void initcalbuf(int);
void setcalbuf(int, int, int, BOOL);
void printcalbuf(int);
void putday(char *, int, int, BOOL);
int tomonth(int *, int *, int *);
int getmonth(char *);
int dayweek(int, int, int);
int mongth(int, int);
int getcalopt(char *, int *, int *);
int readcaltab(void);
int get2digit(char *);
char *center(char *, int);
void terpri(int, FILE *);
void spaces(int, FILE *);
void usage(void);
int thisyear=0, thismonth=0, today=0;
BOOL isthismonth;

void main(int argc, char **argv)
{
	int ac;
	int year, month;
	int shift = 0;
	int months = 1;
	int mcol = NOVAL;
	char *pt;
	char calenv[9];
	static char *opterr = "%s: $%s: %s: Invalid option\n";
	static char *clkerr = "%s: Clock error\n";
	static char *invalyear = "%s: %d: Invalid year\n";
	static char *invalmonth = "%s: %s: Invalid month\n";

	pt = getenv(LANG);
	if (*(progname = _getprogname(argv[0], PROGNAME)) == 'j' \
			|| (pt != NULL && stricmp(pt, JAPANESE) == 0))
		lang = JAPMODE;
#ifdef MULTILINGUAL
	else if (pt != NULL) {
		if (stricmp(pt, FRENCH) == 0)
			lang = FREMODE;
		else if (stricmp(pt, SPANISH) == 0)
			lang = SPAMODE;
		else if (stricmp(pt, GERMAN) == 0)
			lang = GERMODE;
	}
#endif

#ifndef __MSDOS__
	strcpy(calenv, progname);
	{ char *p = calenv;
	  while (*p) {
	    *p = toupper(*p);
	    p++;
	  }
	}
#else
	strupr(strcpy(calenv, progname));
#endif
	if ((pt = getenv(calenv)) != NULL && *pt == '-') {
		if (getcalopt(pt + 1, &months, &mcol) == ERROR) {
			fprintf(stderr, opterr, progname, calenv, pt);
			usage();
		}
	}
	for (ac = 1; ac < argc && *argv[ac] == '-' && *(argv[ac] + 1); ac++) {
		if (getcalopt(argv[ac] + 1, &months, &mcol) == ERROR)
			usage();
	}
	abbr = (lang == JAPMODE ? 2 : 3);

	if (ac < argc && (*argv[ac] == '-' || *argv[ac] == '+') \
						&& *(argv[ac] + 1) == EOS)
		shift = ((*argv[ac++] == '-') ? -1 : 1);
	tomonth(&thisyear, &thismonth, &today);
	if (ac == argc) {
		if (thisyear==0 && thismonth==0) {
			fprintf(stderr, clkerr, progname);
			exit(SYSERROR);
		}
		year=thisyear, month=thismonth;
	}
	else if (ac + 1 == argc) {
		year = atoi(argv[ac++]);
		month = 1;
		months = NOVAL;
	}
	else if (ac + 2 == argc) {
		if ((month = getmonth(argv[ac])) < 1) {
			fprintf(stderr, invalmonth, progname, argv[ac]);
			exit(SYSERROR);
		}
		++ac;
		year = atoi(argv[ac++]);
	}
	else
		usage();

	if (shift) {
		if ((month += shift) > 12) {
			month = 1;
			++year;
		}
		else if (month < 1) {
			month = 12;
			--year;
		}
	}
	if (year < 1 || 9999 < year) {
		fprintf(stderr, invalyear, progname, year);
		exit(SYSERROR);
	}

	if (!t_opt && readcaltab() == ERROR)
		t_opt = TRUE;
	if (t_opt)
		linelen = C1WIDTH * 7 - 1;
	if (mcol == NOVAL)
		mcol = (t_opt ? MAXCOL : DEFCOL);
	headertype = (t_opt ? 1 : 0);

	setbuf(stdout, obuf);

	cal(year, month, months == 0 ? mcol : months, mcol);

	exit(NOERROR);
}


void cal(int year, register int month, register int count, int mcol)
{
	BOOL year_flag = TRUE;
	register int col;
	char yearbuf[MAXWIDTH];

	if (count == NOVAL) {
		terpri(1, stdout);
		sprintf(yearbuf, (lang == JAPMODE) ? YEARJFMT : YEAREFMT, year);
		fputs(center(yearbuf, linelen * mcol + MARGIN * (mcol - 1)), stdout);
		terpri(2, stdout);
		count = 12;
		year_flag = FALSE;
	}

	for (col = 0; col < mcol; col++)
		strcpy(calbuf[col][ROWHEADER], header[lang][headertype]);

	while (count > 0) {
		for (col = 0; col < mcol && count > 0; col++, month++, count--) {
			if (month > 12) {
				if (++year > 9999) {
					count = 0;
					break;
				}
				month = 1;
			}
			initcalbuf(col);
			setcalbuf(col, year, month, year_flag);
		}
		printcalbuf(col);
		if (count > 0)
			terpri(1, stdout);
	}
}


void initcalbuf(int col)
{
	register int wk, dw;

	for (wk = ROW1WEEK; wk < ROW1WEEK + MONTHWEEK; wk++) {
		for (dw = 0; dw < linelen; dw++)
			calbuf[col][wk][dw] = SPC;
		calbuf[col][wk][dw] = EOS;
	}
}


void setcalbuf(int col, int year, int month, BOOL year_flag)
{
	static BOOL furikae = FALSE;
	int length;
	register int wk, dw, day;
	static int start = -1;

	if (start == -1)
		start = dayweek(year, month, 1);
	length = mongth(year, month);
	isthismonth = ((year==thisyear && month==thismonth) ? TRUE : FALSE);

	if (year_flag) {
		if (lang == JAPMODE)
			sprintf(calbuf[col][ROWMONTH], YEARJFMT " %s",
				year, monthname[lang][month - 1]);
		else
			sprintf(calbuf[col][ROWMONTH], "%s " YEAREFMT,
				monthname[lang][month - 1], year);
	}
	else
		strcpy(calbuf[col][ROWMONTH], monthname[lang][month - 1]);
	center(calbuf[col][ROWMONTH], linelen);

	for (wk = ROW1WEEK, dw = start, day = 1;
			day <= length && wk < ROW1WEEK + MONTHWEEK; wk++) {
		for (dw = adjdw(dw); dw < 7 && day <= length; dw++, day++) {
			putday(calbuf[col][wk], dw, day,
				dw == 0 || isholiday(month, day) || furikae);
			if (dw == 0 && isholiday(month, day))
				furikae = TRUE;
			else
				furikae = FALSE;
			if (year == 1752 && month == 9 && day == 2)
				day += (JURIUSGAP - 1);
		}
	}
	start = dw;
}


void printcalbuf(int col)
{
	register int i, j;

	for (j = 0; j < CALBUFROW; j++) {
		for (i = 0; i < col; i++) {
			fputs(calbuf[i][j], stdout);
			if (i < col - 1)
				spaces(MARGIN, stdout);
			else
				terpri(1, stdout);
		}
	}
}


void putday(char *buf, int dw, int day, BOOL holiday)
{
	register char *pt;

	if (t_opt)
		pt = buf + dw * C1WIDTH + 1;
	else
		pt = buf + dw * C2WIDTH + 2;
	if (day >= 10)
		*(pt - 1) = day / 10 + '0';
	*pt = day % 10 + '0';
	if (t_opt)
	  return;
	if (holiday) {
	    if (isthismonth && today==day) {
		*(pt - 2) = '[';
		*(pt + 1) = ']';
	    } else {
		*(pt - 2) = '(';
		*(pt + 1) = ')';
	    }
	} else if (isthismonth && today==day) {
	    *(pt - 2) = '=';
	    *(pt + 1) = '=';
	}
}


int tomonth(int *year, int *month, int *today)
{
	time_t gmt;
	struct tm *dtm;

	if (time(&gmt) == (time_t)-1)
		return (ERROR);
	dtm = localtime(&gmt);
	*year = dtm->tm_year + 1900;
	*month = dtm->tm_mon + 1;
	*today = dtm->tm_mday;
	return (NOERROR);
}


int getmonth(char *str)
{
	int i;
	int month = 0;

	if (isdigit(*str))
		month = atoi(str);
	else {
		for (i = 0; i < 12; i++) {
			if (strnicmp(monthname[lang][i], str, abbr) == 0) {
				month = i + 1;
				break;
			}
		}
	}

	if (1 <= month && month <= 12)
		return (month);
	else
		return (0);
}


int dayweek(int year, int month, int day)
{
	register int i, days;

	if (year < 1752 || (year == 1752 && month < 10))
		days = (year - 1) + (year - 1) / 4 + 5;
	else
		days = (year - 1) + (year - 1) / 4 - (year - 1) / 100 + (year - 1) / 400;
	for (i = 1; i < month; i++)
		days += mongth(year, i);
	days += day;

	return (days % 7);
}


int mongth(int year, int month)
{
	register int days;
	static int mdays[12] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

	if (1 <= month && month <= 12) {
		days = mdays[month - 1];
		if (year < 1753 && month == 2 && year % 4 == 0)
			++days;
		else if (month == 2 && year % 4 == 0 \
							&& (year % 100 != 0 || year % 400 == 0))
			++days;
		return (days);
	}		
	else
		return (ERROR);
}


int getcalopt(char *opts, int *months, int *mcol)
{
	while (*opts) {
		if (*opts == 'e')
			lang = ENGMODE;
		else if (*opts == 'j')
			lang = JAPMODE;
		else if (*opts == 't')
			t_opt = TRUE;
		else if (*opts == 'm') {
			if (isdigit(*(opts + 1))) {
				if ((*months = atoi(opts + 1)) < 0)
					*months = 0;
				break;
			}
			else
				*months = 0;
		}
		else if (isdigit(*opts)) {
			*mcol = *opts - '0';
			if (*mcol < 1)
				*mcol = 1;
			else if (*mcol > MAXCOL)
				*mcol = MAXCOL;
		}
		else
			return (ERROR);
		++opts;
	}

	return (NOERROR);
}


int readcaltab(void)
{
	register int month, day;
	char buf[READBUFSIZE];
	char *caltab;
	register char *pt;
	static char *invaldata = "%s: %s: %s: Invalid data, ignored\n";
	static char *readerr = "%s: %s: Read error\n";
	FILE *fp;

	if ((caltab = getenv(CALTABENV)) == NULL)
		caltab = DEFCALTAB;

	if ((fp = fopen(caltab, "r")) == NULL)
		return (ERROR);
	while (fgets(buf, READBUFSIZE, fp) != NULL) {
		if (*(pt = strtail(buf)) == NL)
			*pt = EOS;
		pt = buf;
		while (*pt == SPC || *pt == TAB)
			++pt;
		if (*pt == COMMENT || *pt == EOS)
			continue;
		if ((month = get2digit(pt)) == ERROR \
					|| (day = get2digit(pt + 2)) == ERROR)
			fprintf(stderr, invaldata, progname, caltab, pt);
		else
			setholiday(month, day);
	}
	if (ferror(fp) || fclose(fp)) {
		fprintf(stderr, readerr, progname, caltab);
		return (ERROR);
	}

	return (NOERROR);
}


int get2digit(char *str)
{
	register int n;

	if (isdigit(*str))
		n = (*str - '0') * 10;
	else
		return (ERROR);
	if (isdigit(*(str + 1)))
		n += *(str + 1) - '0';
	else
		return (ERROR);
	return (n);
}


char *center(char *str, int width)
{
	int pad;
	register int len;
	char *orgstr;
	register char *src, *dest;

	if ((len = strlen(str)) >= width)
		return (str);
	if ((src = orgstr = strdup(str)) == NULL)
		return (str);

	pad = (width - len) / 2;
	len = 0;
	dest = str;
	while (len < pad) {
		*dest++ = SPC;
		++len;
	}
	while (*src) {
		*dest++ = *src++;
		++len;
	}
	while (len < width) {
		*dest++ = SPC;
		++len;
	}
	*dest = EOS;

	free(orgstr);

	return(str);
}


void terpri(int n, FILE *fp)
{
	while (n--)
		fputc(NL, fp);
}


void spaces(int n, FILE *fp)
{
	while (n--)
		fputc(SPC, fp);
}


void usage(void)
{
	fprintf(stderr, "Usage: %s [-#ejmt] [-|+] [[month] year]\n", progname);
	exit(USERERROR);
}


/*
$Log:	cal.c $
 * Revision 1.10  94/07/07  23:54:52  Sugi
 * ̾ǻꤷݤΥХ
 * 
 * Revision 1.9  92/09/12  18:08:58  Sugi
 * ѥ롦ץ MULTILINGUAL ɲ
 * ѥ롦ץ NOALIAS ɲ
 * -m ץθ夬ͤǤʤν
 * Ķѿ LANG ȤΥХ
 * 
 * revision 1.8 Sugi 92/08/20 22:16:09
 * Ϥˤäν꤬Τ
 * cast θ(ʤ LSI C-86 Ǥ̤)
 * 
 * revision 1.7 Sugi 92/08/18 23:48:08
 * caltab ¸ߤʤΥХ
 * 
 * revision 1.6 Sugi 92/08/15 23:50:52
 * caltab 顼ΥХ
 * 
 * revision 1.5 Sugi 92/08/02 17:45:38
 * ̾ǻǤ褦ˤ
 * Ǥ褦ˤ
 * 
 * revision 1.4 Sugi 92/07/28 01:54:58
 * Ķѿ lang 򸫤褦ˤ
 * 
 * revision 1.3 Sugi 92/02/28 19:55:40
 * -m ץΥХ
 * Ķѿ cal ǥץǽ
 * 
 * revision 1.2 Sugi 92/02/26 02:18:40
 * -#, -m# ץɲ
 * 
 * revision 1.1 Sugi 92/02/19 02:27:35
 * Initial revision
 * 
 */
