/*
 * Copyright 2001 Wasabi Systems, Inc.
 * All rights reserved.
 *
 * Written by Frank van der Linden for Wasabi Systems, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed for the NetBSD Project by
 *      Wasabi Systems, Inc.
 * 4. The name of Wasabi Systems, Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASABI SYSTEMS, INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASABI SYSTEMS, INC
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/filedesc.h>
#include <sys/file.h>
#include <sys/filio.h>
#include <sys/select.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/mbuf.h>
#include <sys/poll.h>
#include <sys/proc.h>
#include <sys/signalvar.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/sysctl.h>
#include <sys/systm.h>
#include <sys/ttycom.h>
#include <sys/uio.h>
#include <sys/vnode.h>
#include <sys/exec.h>
#include <sys/lkm.h>

#include <net/bpf.h>
#include <net/if.h>
#include <net/if_ether.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <net/route.h>

#include <netinet/in.h>

#ifdef INET
#include <netinet/if_inarp.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <compat/linux/common/linux_ioctl.h>
#include <compat/linux/common/linux_sockio.h>

#include "if_hubvar.h"
#include "if_hub.h"

#if __NetBSD_Version__ > 105009900
#include <uvm/uvm_extern.h>
#include <uvm/uvm_param.h>
#else
static int ether_ioctl(struct ifnet *, u_long, caddr_t);
#include <vm/vm.h>
#endif

#define HUBDEBUG  if (hub_debug) printf

int if_hub_lkmentry(struct lkm_table *, int, int);

static int hub_open(dev_t dev, int oflags, int devtype, struct proc *p);
static int hub_close(dev_t dev, int cflags, int devtype, struct proc *p);
static int hub_ioctl(dev_t dev, u_long cmd, caddr_t data, int flags,
		 struct proc *p);
static int hub_read(dev_t, struct uio *, int);
static int hub_write(dev_t, struct uio *, int);
static int hub_poll(dev_t, int, struct proc *);

static int hub_ifinit(struct ifnet *);
static int hub_handle(struct lkm_table *, int);
static void hub_ifstart(struct ifnet *);
static int hub_ifioctl(struct ifnet *, u_long, caddr_t);

static void port_destroy(struct hubdev_softc *, int);
static int port_create(struct hubdev_softc *, struct hubport_softc **);
static void port_netifattach(struct hubport_softc *, const char *);
static int port_pktmatch(struct hubport_softc *, struct mbuf *);
static struct hubport_softc *port_allocate(struct hubdev_softc *);
static void port_deallocate(struct hubport_softc *);

static int hub_sendchain(struct hubport_softc *, struct mbuf *);
static int hub_fake_clonedev(dev_t, int, struct proc *);



static struct cdevsw hub_dev = {
	hub_open, hub_close, 
	hub_read, hub_write,
	hub_ioctl, (dev_type_stop((*))) enodev, 0,
	hub_poll, (dev_type_mmap((*))) enodev, 0
};

static int hub_refcnt = 0;
static int hub_debug = 0;
static struct hubdev_softc *hub_scs[MAXHUBDEVS];

#if __NetBSD_Version__ >= 106080000
MOD_DEV("vmnet", "vmnet", NULL, -1, &hub_dev, -1)
#else
MOD_DEV("vmnet", LM_DT_CHAR, -1, &hub_dev)
#endif

int
if_hub_lkmentry(struct lkm_table *lkmtp, int cmd, int ver)
{
	DISPATCH(lkmtp, cmd, ver, hub_handle, hub_handle, hub_handle)
}

static int 
hub_handle(struct lkm_table *lkmtp, int cmd)
{
	int error = 0, i, j;
	struct hubdev_softc *sc;
#if __NetBSD_Version__ >= 105230000 || (__NetBSD_Version__ >= 105000100 && __NetBSD_Version__ < 105010000)
	struct ifnet *ifp;
#endif
	
	switch (cmd) {
	case LKM_E_LOAD:
#if __NetBSD_Version__ >= 105230000 || (__NetBSD_Version__ >= 105000100 && __NetBSD_Version__ < 105010000)
 		ifp = (struct ifnet *)malloc(sizeof(*ifp), M_MRTABLE, M_WAITOK);
		ifp->if_mtu = ETHERMTU;
		if (HUBMAXPKT(ifp) > MCLBYTES) {
#else
		if (HUBMAXPKT > MCLBYTES) {
#endif
			printf("if_hub: HUBMAXPKT > MCLBYTES\n");
			return EIO;
		}
		if (lkmexists(lkmtp)) 
			return EEXIST;
		break;
		
	case LKM_E_UNLOAD:
		if (hub_refcnt > 0)
			return (EBUSY);

		for (i = 0; i < MAXHUBDEVS; i++) {
			sc = hub_scs[i];
			if (sc == NULL)
				continue;
			for (j = 0; j <= sc->sc_maxport; j++)
				port_destroy(sc, j);
			free(sc, M_DEVBUF);
		}
		break;
		
	case LKM_E_STAT:
		break;

	default:
		error = EIO;
		break;
	}
	return error;
}

static void
port_destroy(struct hubdev_softc *hubsc, int num)
{
	struct hubport_softc *portsc;

	portsc = hubsc->sc_ports[num];
	if (portsc == NULL)
		return;
	hubsc->sc_ports[num] = NULL;
	if (num == hubsc->sc_maxport)
		hubsc->sc_maxport--;

	port_deallocate(portsc);
}

static void
port_deallocate(struct hubport_softc *portsc)
{
	int s;

	if (portsc->port_flags & PORT_HOSTIF)  {
		s = splnet();
		ether_ifdetach(&portsc->port_if);
		if_detach(&portsc->port_if);
		splx(s);
	}
	FREE(portsc, M_DEVBUF);
}

static struct hubport_softc *
port_allocate(struct hubdev_softc *hubsc)
{
	struct ifnet *ifp;
	struct hubport_softc *portsc;

	portsc = malloc(sizeof *portsc, M_DEVBUF, M_WAITOK);
	if (portsc == NULL)
		return NULL;

	memset(portsc, 0, sizeof *portsc);
	memset(portsc->port_addr, 0, sizeof portsc->port_addr);

	/*
	 * Set first three octets to the VMware OUI. The rest
	 * needs to be set later via a LINUX_VMWARE_SIOCSIFADDR
	 * ioctl call.
	 */
	portsc->port_addr[0] = VMX86_OUI0;
	portsc->port_addr[1] = VMX86_OUI1;
	portsc->port_addr[2] = VMX86_OUI2;

	portsc->port_hub = hubsc;

	ifp = &portsc->port_if;
	ifp->if_softc = portsc;

	ifp->if_start = hub_ifstart;
	ifp->if_ioctl = hub_ifioctl;
#if __NetBSD_Version__ > 105009900
	ifp->if_init = hub_ifinit;
	ifp->if_stop = if_nullstop;
#endif
	ifp->if_mtu = ETHERMTU;
	/*
	 * XXX the Linux vmnet seems to set IFF_RUNNING straight away,
	 * so mimic the behaviour.
	 */
	ifp->if_flags = (IFF_BROADCAST|IFF_SIMPLEX|IFF_MULTICAST|IFF_RUNNING);
	ifp->if_snd.ifq_maxlen = IFQ_MAXLEN;
	portsc->port_rcvq.ifq_maxlen = IFQ_MAXLEN;

	/*
	 * Do not attach this interface. Only the real interface
	 * (HOSTIF flag set), will get attached through the
	 * LINUX_VMWARE_SIOCNETIF ioctl.
	 */

	return portsc;
}


static int
port_create(struct hubdev_softc *sc, struct hubport_softc **portpp)
{
	struct ifnet *ifp;
	struct hubport_softc *portsc;
	int i;

	for (i = 0; i < MAXHUBPORTS; i++)
		if (sc->sc_ports[i] == NULL)
			break;
	if (i == MAXHUBPORTS)
		return EBUSY;

	portsc = port_allocate(sc);
	if (portsc == NULL)
		return ENOMEM;
	sc->sc_ports[i] = portsc;
	if (i >= sc->sc_maxport)
		sc->sc_maxport = i;
	portsc->port_dev = MAKEPORTDEV(sc->sc_dev, HUBUNIT(sc->sc_dev), i);
	snprintf(portsc->port_if.if_xname, sizeof ifp->if_xname, "hub%d", i);

	*portpp = portsc;

	return 0;
}

static void
port_netifattach(struct hubport_softc *portsc, const char *name)
{
	int s;
	struct ifnet *ifp;

	portsc->port_addr[3] = 1;
	portsc->port_addr[4] = portsc->port_addr[5] = 0;

	ifp = &portsc->port_if;
	snprintf(ifp->if_xname, sizeof ifp->if_xname, "%s", name);
	portsc->port_flags |= PORT_HOSTIF;
	if_attach(ifp);
	s = splnet();
	ether_ifattach(ifp, portsc->port_addr);
	splx(s);
#if NBPFILTER > 0 && __NetBSD_Version__ < 105009900
	bpfattach(&ifp->if_bpf, ifp, DLT_EN10MB,
	    sizeof(struct ether_header));
#endif
}

static int
port_pktmatch(struct hubport_softc *portsc, struct mbuf *m)
{
	struct ether_header *eh;
	u_int8_t *dst;
	short flags;
	u_int32_t crc;

	eh = mtod(m, struct ether_header *);
	dst = &eh->ether_dhost[0];
	flags = portsc->port_if.if_flags;

	if ((flags & IFF_PROMISC) ||
	    !memcmp(dst, portsc->port_addr, ETHER_ADDR_LEN) ||
	    ((flags & IFF_BROADCAST) &&
	      !memcmp(dst, etherbroadcastaddr, ETHER_ADDR_LEN)))
		return 1;

	if (!ETHER_IS_MULTICAST(dst))
		return 0;

	if (flags & IFF_ALLMULTI)
		return 1;

	/*
	 * This taken from dev/ic/lance.c (VMware emulates le at pci).
	 */
	crc = ether_crc32_le(dst, ETHER_ADDR_LEN);
	crc >>= 26;
	if (portsc->port_ladrf[crc >> 4] & (1 << (crc & 0xf)))
		return 1;

	return 0;
}


static int
hub_open(dev_t dev, int flag, int mode, struct proc *p)
{
	struct hubdev_softc *hubsc;
	struct hubport_softc *portsc;
	int error, unit;

	if (p->p_dupfd >= 0)
		return ENODEV;

	unit = HUBUNIT(dev);

	HUBDEBUG("if_hub: %d opened minor %d hub %d\n", p->p_pid, minor(dev),
	    unit);

	if (unit >= MAXHUBDEVS)
		return (ENXIO);

	if (suser(p->p_ucred, &p->p_acflag) != 0)
		return (EPERM);

	hubsc = hub_scs[unit];
	if (hubsc == NULL) {
		hubsc = malloc(sizeof *hubsc, M_DEVBUF, M_WAITOK);
		if (hubsc == NULL)
			return ENOMEM;
		memset(hubsc, 0, sizeof *hubsc);
		hubsc->sc_dev = dev;
		hub_scs[unit] = hubsc;
	}

	error = port_create(hubsc, &portsc);
	if (error != 0)
		return error;

	hub_refcnt++;

	HUBDEBUG("if_hub: pid %d new port: unit %d, port %d\n",
	    p->p_pid, HUBUNIT(portsc->port_dev),
	    HUBPORT(portsc->port_dev));

	error = hub_fake_clonedev(portsc->port_dev, flag, p);
	if (error != 0 && p->p_dupfd < 0)
		port_destroy(hubsc, HUBPORT(portsc->port_dev));

	return error;
}


/*
 * hub_close
 *
 * close the device - mark i/f down & delete routing info
 */
static int
hub_close(dev_t dev, int flags, int mode, struct proc *p)
{
	int s, unit, port;
	struct hubdev_softc *hubsc;
	struct hubport_softc *portsc;
	struct mbuf *m;

	HUBDEBUG("if_hub: close hub %d unit %d by pid %d\n",
	    HUBUNIT(dev), HUBPORT(dev), p->p_pid);
	/*
	 * The 2 cases below shouldn't ever happen.
	 */
	unit = HUBUNIT(dev);
	if (unit >= MAXHUBDEVS || (hubsc = hub_scs[HUBUNIT(dev)]) == NULL) {
		HUBDEBUG("if_hub: close: illegal unit %d??\n", unit);
		return ENXIO;
	}

	port = HUBPORT(dev);
	if (port >= MAXHUBPORTS || (portsc = hubsc->sc_ports[port]) == NULL) {
		HUBDEBUG("if_hub: close: illegal port %d??\n", port);
		return ENXIO;
	}

	s = splnet();
	if (portsc->port_rcvq.ifq_len > 0) {
		do {
			IF_DEQUEUE(&portsc->port_rcvq, m);
			if (m != NULL)
				m_freem(m);
		} while (m != NULL);
	}
	splx(s);

	port_destroy(hubsc, port);

	hub_refcnt--;
	if (hub_refcnt < 0) {
		hub_refcnt = 0;
		printf("if_hub: refcnt < 0 ??\n");
	}

	HUBDEBUG("if_hub: hub %d port %d closed by %d\n", unit, port, p->p_pid);

	return (0);
}


static int
hub_ifinit(struct ifnet *ifp)
{
	ifp->if_flags |= IFF_RUNNING;
	ifp->if_flags &= ~IFF_OACTIVE;

	hub_ifstart(ifp);

	return 0;
}


static int
hub_ifioctl(struct ifnet *ifp, u_long cmd, caddr_t data)
{
	int error, s;

	s = splnet();
	error = ether_ioctl(ifp, cmd, data);
	splx(s);
	return error;
}

/*
 * XXXX - poor man's device cloning.
 */
int
hub_fake_clonedev(dev_t dev, int flag, struct proc *p)
{
	struct file *fp;
	int error, fd;
	struct vnode *vp;

	if (flag & (O_EXLOCK | O_SHLOCK))
		/* XXX */
		return EINVAL;

	error = falloc(p, &fp, &fd);
	if (error != 0)
		return error;
	error = cdevvp(dev, &vp);
	if (error != 0)
		return error;

	if (flag & FWRITE)
		vp->v_writecount++;

	fp->f_flag = flag & FMASK;
	fp->f_type = DTYPE_VNODE;
	fp->f_ops = &vnops;
	fp->f_data = (caddr_t)vp;
#if __NetBSD_Version__ >= 105230000 || (__NetBSD_Version__ >= 105000100 && __NetBSD_Version__ < 105010000)
#ifdef FILE_SET_MATURE
	FILE_SET_MATURE(fp);
#endif
#endif
	FILE_UNUSE(fp, p);

	p->p_dupfd = fd;

	return ENXIO;
}

/*
 * Send an ethernet packet to all who want it. For the host interface,
 * do it the official way, using the interface's input routine. Otherwise
 * just copy it, we're a virtual hub, no interpretation wanted.
 */
static int
hub_sendchain(struct hubport_softc *portsc, struct mbuf *m)
{
	struct hubdev_softc *hubsc;
	struct ifnet *ifp, *ifp2;
	struct mbuf *m2;
	struct hubport_softc *psc2;
	int drops1, drops2, s, sent, i;
	struct proc *p;

	ifp = &portsc->port_if;
	hubsc = portsc->port_hub;

	for (i = 0; i <= hubsc->sc_maxport; i++) {
		psc2 = hubsc->sc_ports[i];
		if (psc2 == NULL || psc2 == portsc)
			continue;
		if (!port_pktmatch(psc2, m)) {
			continue;
		}
		if ((psc2->port_if.if_flags & (IFF_UP | IFF_RUNNING)) !=
		    (IFF_UP | IFF_RUNNING)) {
			continue;
		}
		sent = 0;
		ifp2 = &psc2->port_if;
		if (psc2->port_flags & PORT_HOSTIF) {
			m2 = m_copym(m, 0, M_COPYALL, M_DONTWAIT);
			if (m2 == NULL)
				continue;
			m2->m_pkthdr.rcvif = ifp2;
			/*
			 * XXX check if the packet was accepted
			 * by looking at the dropcount.
			 */
#if NBPFILTER > 0
			if (ifp2->if_bpf)
				bpf_mtap(ifp2->if_bpf, m2);
#endif
			s = splnet();
			drops1 = ifp2->if_snd.ifq_drops;
			splx(s);

			ifp2->if_input(ifp2, m2);
			ifp2->if_ipackets++;

			s = splnet();
			drops2 = ifp2->if_snd.ifq_drops;
			splx(s);

			sent = (drops1 == drops2);
		} else {
			if (!IF_QFULL(&psc2->port_rcvq)) {
				m2 = m_copym(m, 0, M_COPYALL, M_DONTWAIT);
				if (m2 == NULL)
					continue;
				IF_ENQUEUE(&psc2->port_rcvq, m2);
				sent = 1;
				psc2->port_if.if_ipackets++;
			} else {
				psc2->port_rcvq.ifq_drops++;
			}
		}
		if (sent != 0) {
			if (psc2->port_flags & PORT_RWAIT) {
				psc2->port_flags &= ~PORT_RWAIT;
				wakeup((caddr_t)psc2);
			}
			if ((psc2->port_flags & PORT_ASYNC) != 0) {
				if (psc2->port_sigio > 0)
					gsignal(psc2->port_sigio, SIGIO);
				else {
					p = pfind(-psc2->port_sigio);
					if (p != NULL)
						psignal(p, SIGIO);
				}
			}
			selwakeup(&psc2->port_rsel);
		}
	}
	return 0;
}

/*
 * XXX only deal with one packet at the time, since that is
 * what readers and writers expect.
 */
static void
hub_ifstart(struct ifnet *ifp)
{
	struct hubport_softc *portsc = ifp->if_softc;
	int s;
	struct mbuf *m;

	HUBDEBUG("if_hub: hostif %s start output\n", ifp->if_xname);

	/*
	 * Can this happen?
	 */
	if (ifp->if_snd.ifq_len == 0)
		return;

	s = splnet();
	ifp->if_flags |= IFF_OACTIVE;

#if __NetBSD_Version__ > 105009900
	IFQ_DEQUEUE(&ifp->if_snd, m);
#else
	IF_DEQUEUE(&ifp->if_snd, m);
#endif

	hub_sendchain(portsc, m);
#if NBPFILTER > 0
	if (ifp->if_bpf)
		bpf_mtap(ifp->if_bpf, m);
#endif
	m_freem(m);

	ifp->if_opackets++;

	ifp->if_flags &= ~IFF_OACTIVE;

	splx(s);
}


static int
hub_ioctl(dev_t dev, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct hubport_softc *portsc, *newport;
	struct hubdev_softc *hubsc;
	struct ifnet *ifp;
	int s, error, i, port;
	struct ioctl_pt *pt;
	unsigned short f;
	struct linux_vmware_setmacaddr_args sma;
	char name[8];
	u_int8_t addr[ETHER_ADDR_LEN], *addr2;
	struct timeval tv;
	int zero = 0;
	u_long val;

	HUBDEBUG("if_hub: ioctl %lx on hub %d port %d by pid %d\n",
	    cmd, HUBUNIT(dev), HUBPORT(dev), p->p_pid);

	hubsc = hub_scs[HUBUNIT(dev)];
	if (hubsc == NULL)
		return ENXIO;
	port = HUBPORT(dev);
	portsc = hubsc->sc_ports[port];
	if (portsc == NULL)
		return ENXIO;

	ifp = &portsc->port_if;

	switch (cmd) {
	/*
	 * XXX - the FIO* and TIOC* stuff was taken from if_tun,
	 * and might not be needed.
	 */
	case FIONBIO:
		portsc->port_flags |= PORT_NBIO;
		break;

	case FIOASYNC:
		s = splnet();
		if (*(int *)data)
			portsc->port_flags |= PORT_ASYNC;
		else
			portsc->port_flags &= ~PORT_ASYNC;
		splx(s);
		break;

	case FIONREAD:
		s = splnet();
		if (portsc->port_rcvq.ifq_head) {
			struct mbuf *mb = portsc->port_rcvq.ifq_head;

			for(*(int *)data = 0;mb != NULL;mb = mb->m_next)
				*(int *)data += mb->m_len;
		} else
			*(int *)data = 0;
		splx(s);
		break;

	case TIOCSPGRP:
		portsc->port_sigio = *(int *)data;

	case TIOCGPGRP:
		*(int *)data = portsc->port_sigio;
		return (0);

	/*
	 * VMware/VMnet port ioctl's
	 * Some of these overload SIOC* in a bad way (different args),
	 * so they get here via the PTIOCLINUX passthrough ioctl.
	 */

	case PTIOCLINUX:
		pt = (struct ioctl_pt *)data;
		switch (pt->com) {
		case LINUX_SIOCGIFFLAGS:
			val = (u_long)ifp->if_flags;
			return copyout(&val, pt->data, sizeof val);
		case LINUX_SIOCSIFFLAGS:
			f = (u_short)(u_long)pt->data;

			f &= ~IFF_CANTCHANGE;
			f |= IFF_RUNNING;	/* VMWare never clears this */
			s = splnet();
			ifp->if_flags = f | (ifp->if_flags & IFF_CANTCHANGE);
			splx(s);
			break;
		case LINUX_SIOCGIFADDR:
			return copyout(portsc->port_addr, pt->data,
			    sizeof portsc->port_addr);
		case LINUX_SIOCSIFADDR:
			error = copyin(pt->data, addr, sizeof addr);
			if (error != 0)
				return error;
			if (addr[0] != VMX86_OUI0 || addr[1] != VMX86_OUI1 ||
			    addr[2] != VMX86_OUI2)
				return EINVAL;

			/*
			 * XXX VMWare also has an IP-based MAC addr allocation
			 * policy, which is not implemented here.
			 */
			if (!(addr[3] & (VMX86_MAC_RANDOM | VMX86_MAC_IPBASED)))
				memcpy(portsc->port_addr, addr, sizeof addr);
			else {
				/*
				 * Generate a new address, and see if it's
				 * not already in use. Given that there
				 * are just 15 hosts on one hub, this
				 * should find a good match soon.
				 */
				while (1) {
					for (i = 0; i <= hubsc->sc_maxport; i++)
					{
						if (hubsc->sc_ports[i] == NULL)
							continue;
						addr2 =
						  hubsc->sc_ports[i]->port_addr;
						if (!memcmp(addr, addr2,
							    sizeof addr))
							break;
					}
					if (i > hubsc->sc_maxport) {
						break;
					}
					microtime(&tv);
					addr[3] = ((tv.tv_usec >> 8) & 0xff)
							 & ~VMX86_MAC_PREFIX;
					addr[4] = tv.tv_usec & 0xff;
					addr[5] = (tv.tv_usec >> 16) & 0xff;
				}
				memcpy(portsc->port_addr, addr, sizeof addr);
			}
			break;
		case LINUX_SIOCGIFBR:
			error = copyout(&zero, pt->data, sizeof zero);
			if (error != 0)
				return error;
			break;
		case LINUX_VMWARE_SIOCPORT:
			newport = port_allocate(hubsc);
			if (newport == NULL)
				return ENOMEM;
			newport->port_dev = portsc->port_dev;
			strcpy(newport->port_if.if_xname,
			    portsc->port_if.if_xname);
			port_deallocate(portsc);
			hubsc->sc_ports[port] = newport;
			break;
		case LINUX_VMWARE_SIOCSLADRF:
			error = copyin(pt->data, portsc->port_ladrf,
			    sizeof portsc->port_ladrf);
			if (error != 0)
				return error;
			break;
		case LINUX_VMWARE_SIOCNETIF:
			error = copyin(pt->data, name, sizeof name);
			if (error != 0)
				return error;
			port_netifattach(portsc, name);
			break;
		case LINUX_VMWARE_SETMACADDR:
			error = copyin(pt->data, &sma, sizeof sma);
			if (error != 0)
				return error;
			if (sma.version != 1)
				return EINVAL;
			memcpy(portsc->port_addr, sma.addr,
			    sizeof sma.addr);
			break;
		default:
			return ENOTTY;
		}
		break;

	default:
		return ENOTTY;
	}
	return (0);
}


/*
 * hub_read (based on tunread)
 *
 * the cdevsw read interface - reads a packet at a time, or at
 * least as much of a packet as can be read
 */
static int
hub_read(dev_t dev, struct uio *uio, int flag)
{
	struct hubdev_softc *hubsc;
	struct hubport_softc *portsc;
	struct ifnet *ifp;
	struct ifqueue *ifq;
	struct mbuf *m, *m0;
	int error = 0, len, s;

	hubsc = hub_scs[HUBUNIT(dev)];
	if (hubsc == NULL)
		return ENXIO;
	portsc = hubsc->sc_ports[HUBPORT(dev)];
	if (portsc == NULL)
		return ENXIO;

	HUBDEBUG("if_hub: read on hub %d port %d by pid %d\n",
	    HUBUNIT(dev), HUBPORT(dev), curproc->p_pid);

	if (portsc->port_flags & PORT_HOSTIF) {
		HUBDEBUG("if_hub: read on real interface??\n");
		return ENXIO;
	}

	portsc->port_flags &= ~PORT_RWAIT;
	ifp = &portsc->port_if;
	ifq = &portsc->port_rcvq;

	do {
		s = splnet();
		IF_DEQUEUE(ifq, m0);
		splx(s);

		if (m0 == NULL) {
			if (flag & IO_NDELAY ||
			    (portsc->port_flags & PORT_NBIO))
				return (EWOULDBLOCK);
			
			portsc->port_flags |= PORT_RWAIT;
			error = tsleep((caddr_t)portsc, PCATCH |(PZERO + 1),
			    "hubrd", 0);
			if (error)
				return (error);
		}
	} while (m0 == NULL);

	while ((m0 != NULL) && (uio->uio_resid > 0) && (error == 0)) {
		len = min(uio->uio_resid, m0->m_len);
		if (len == 0)
			break;

		error = uiomove(mtod(m0, caddr_t), len, uio);
		MFREE(m0, m);
		m0 = m;
	}

	if (m0 != NULL) {
		HUBDEBUG("if_hub: hub %d port %d drop mbuf on read by pid %d\n",
		    HUBUNIT(dev), HUBPORT(dev), curproc->p_pid);
		m_freem(m0);
	}

	return (error);
}


/*
 * Write no more than one packet at a time. The code assumes that
 * one packet will fit in an mbuf cluster. Therefore, HUBMAXPKT
 * has to be smaller than MCLBYTES (this is checked when loading
 * the LKM). Since we're only dealing with plain ethernet, this is
 * always the case.
 */
static int
hub_write(dev_t dev, struct uio *uio, int flag)
{
	struct hubdev_softc *hubsc;
	struct hubport_softc *portsc;
	struct mbuf *m;
	int error;
#if __NetBSD_Version__ >= 105230000 || (__NetBSD_Version__ >= 105000100 && __NetBSD_Version__ < 105010000)
	struct ifnet *ifp;
#endif

	hubsc = hub_scs[HUBUNIT(dev)];
	if (hubsc == NULL)
		return ENXIO;
	portsc = hubsc->sc_ports[HUBPORT(dev)];
	if (portsc == NULL)
		return ENXIO;

	HUBDEBUG("if_hub: write on hub %d port %d len %d by pid %d\n",
	    HUBUNIT(dev), HUBPORT(dev), uio->uio_resid, curproc->p_pid);

	if (uio->uio_resid == 0)
		return (0);

#if __NetBSD_Version__ >= 105230000 || (__NetBSD_Version__ >= 105000100 && __NetBSD_Version__ < 105010000)
	ifp = &portsc->port_if;
	if ((uio->uio_resid < HUBMINPKT) || (uio->uio_resid > HUBMAXPKT(ifp))) {
#else
	if ((uio->uio_resid < HUBMINPKT) || (uio->uio_resid > HUBMAXPKT)) {
#endif
		HUBDEBUG("if_hub: write: illegal length %d\n", uio->uio_resid);
		return EIO;
	}

	/* get a header mbuf */
	MGETHDR(m, M_DONTWAIT, MT_DATA);
	if (m == NULL)
		return (ENOBUFS);
	MCLGET(m, M_DONTWAIT);
	if (!(m->m_flags & M_EXT)) {
		m_freem(m);
		return ENOBUFS;
	}

	m->m_pkthdr.len = uio->uio_resid;
	m->m_len = uio->uio_resid;

	error = uiomove(mtod(m, caddr_t), uio->uio_resid, uio);
	if (error != 0) {
		portsc->port_if.if_ierrors++;
		m_freem(m);
		return (error);
	}

	hub_sendchain(portsc, m);

	m_freem(m);

	return (0);
}


/*
 * hub_poll
 *
 * the poll interface, this is only useful on reads
 * really. the write detect always returns true, write never blocks
 * anyway, it either accepts the packet or drops it
 */
static int
hub_poll(dev_t dev, int events, struct proc *p)
{
	struct hubdev_softc *hubsc;
	struct hubport_softc *portsc;
	int s, revents = 0;

	hubsc = hub_scs[HUBUNIT(dev)];
	if (hubsc == NULL)
		return ENXIO;
	portsc = hubsc->sc_ports[HUBPORT(dev)];
	if (portsc == NULL)
		return ENXIO;

	HUBDEBUG("if_hub: poll on hub %d port %d by pid %d\n",
	    HUBUNIT(dev), HUBPORT(dev), p->p_pid);

	s = splnet();
	if (events & (POLLIN | POLLRDNORM)) {
		if (portsc->port_rcvq.ifq_len > 0)
			revents |= (events & (POLLIN | POLLRDNORM));
		else
			selrecord(p, &portsc->port_rsel);
	}

	if (events & (POLLOUT | POLLWRNORM))
		revents |= (events & (POLLOUT | POLLWRNORM));

	splx(s);
	return (revents);
}


#if __NetBSD_Version__ <= 105009900
/*
 * Common ioctls for Ethernet interfaces.  Note, we must be
 * called at splnet().
 */
static int
ether_ioctl(struct ifnet *ifp, u_long cmd, caddr_t data)
{
	struct ethercom *ec = (void *) ifp;
	struct ifreq *ifr = (struct ifreq *)data;
	struct ifaddr *ifa = (struct ifaddr *)data;
	int error = 0;

	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			if ((error = hub_ifinit(ifp)) != 0)
				break;
			arp_ifinit(ifp, ifa);
			break;
#endif /* INET */
#ifdef NS
		case AF_NS:
		    {
			struct ns_addr *ina = &IA_SNS(ifa)->sns_addr;

			if (ns_nullhost(*ina))
				ina->x_host = *(union ns_host *)
				    LLADDR(ifp->if_sadl);
			else
				memcpy(LLADDR(ifp->if_sadl),
				    ina->x_host.c_host, ifp->if_addrlen);
			/* Set new address. */
			error = (*ifp->if_init)(ifp);
			break;
		    }
#endif /* NS */
		default:
			error = hub_ifinit(ifp);
			break;
		}
		break;

	case SIOCGIFADDR:
		memcpy(((struct sockaddr *)&ifr->ifr_data)->sa_data,
		    LLADDR(ifp->if_sadl), ETHER_ADDR_LEN);
		break;

	case SIOCSIFMTU:
		if (ifr->ifr_mtu > ETHERMTU)
			error = EINVAL;
		else
			ifp->if_mtu = ifr->ifr_mtu;
		break;

	case SIOCSIFFLAGS:
		if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) == IFF_RUNNING) {
			/*
			 * If interface is marked down and it is running,
			 * then stop and disable it.
			 *
			 * Do nothing for if_hub.
			 */
		} else if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) == IFF_UP) {
			/*
			 * If interface is marked up and it is stopped, then
			 * start it.
			 */
			error = hub_ifinit(ifp);
		} else if ((ifp->if_flags & IFF_UP) != 0) {
			/*
			 * Reset the interface to pick up changes in any other
			 * flags that affect the hardware state.
			 */
			error = hub_ifinit(ifp);
		}
		break;

	case SIOCADDMULTI:
	case SIOCDELMULTI:
		error = (cmd == SIOCADDMULTI) ?
		    ether_addmulti(ifr, ec) :
		    ether_delmulti(ifr, ec);
		break;

	default:
		error = ENOTTY;
	}

	return (error);
}
#endif
