/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_hostif[] = "$Id: hostif.c,v 1.5.2.2 2000/06/09 00:21:50 hpreg Exp $";
#else
#define FILECODE "F(303)"
#endif 


/*
 * vmx86p_linux.c
 *
 * interaction with the host OS
 *
 */

#include "driver-config.h"

#include <linux/binfmts.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/malloc.h>

#ifdef KERNEL_2_1
#  ifdef KERNEL_2_3_25
#    include <linux/pagemap.h>	// for kmap/kunmap
#  endif
#  include <asm/uaccess.h>
#  include <asm/fixmap.h>	// for IO_APIC address
#  define memcpy_fromfs		copy_from_user
#  define memcpy_tofs		copy_to_user
#else
#  include <asm/segment.h>
#endif

#ifdef __SMP__
#include <linux/smp.h>
#include <linux/smp_lock.h>
#endif

#include <asm/io.h>
#include <linux/delay.h>

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "vm_asm.h"
#include "modulecall.h"
#include "hostif.h"
#include "memtrack.h"
#include "phystrack.h"



/*
 * First Page Locking strategy
 * ---------------------------
 *
 * An early implementation hacked the lock bit for the purpose of locking
 * memory. This had a couple of advantages:
 *   - the vmscan algorithm would never eliminate mappings from the process address space
 *     easy to assert that things are ok
 *   - it worked with anonymous memory. Basically, vmscan jumps over these pages, their
 *     use count stays high, ....
 *
 * This approach however had a couple of problems:
 *
 *   - it relies on an undocumented interface. (in another words, a total hack)
 *   - it creates deadlock situations if the application gets a kill -9 or
 *     otherwise dies ungracefully. linux first tears down the address space, then
 *     closes file descriptors (including our own device). Unfortunately, this leads
 *     to a deadlock of the process on pages with the lock bit set.
 *
 *     There is a workaround for that, namely to detect that condition using
 *     a linux timer. (ugly)
 *
 * Current Page Locking strategy
 * -----------------------------
 *
 * The current scheme does not use the lock bit, rather it increments the use count
 * on the pages that need to be locked down in memory.
 *
 * The problem is that experiments on certain linux systems (e.g. 2.2.0-pre9) showed
 * that linux somehow swaps out anonymous pages, even with the increased ref counter.
 * Swapping them out to disk is not that big of a deal, but bringing them back
 * to a different location is.  In any case, anonymous pages in linux are not intended
 * to be write-shared (e.g. try to MAP_SHARED /dev/zero).  
 *
 *
 * As a result, the current locking strategy requires that all locked pages are
 * backed by the filesystem, not by swap. For now, we use both mapped files and sys V
 * shared memory. The user application is responsible to cover these cases.
 *
 * About MemTracker and PhysTracker
 * --------------------------------
 *
 * Redundancy is good for now. 
 * 
 * MemTracker is actually required for the NT host version
 * For the linux host, we use both for now
 *
 */

#ifdef CONFIG_HIGHMEM64G
#error "64GB memory configuration is not supported."
#endif 

#ifdef KERNEL_2_1
#define ATOMIC_INCREMENT(_x) atomic_inc(&(_x))
#define ATOMIC_DECREMENT(_x) atomic_dec(&(_x))
#else
#define ATOMIC_INCREMENT(_x) (_x)++
#define ATOMIC_DECREMENT(_x) (_x)--
#endif



#define HOST_LOCK_PFN(_vm,_pfn)     {                   \
   ATOMIC_INCREMENT((mem_map+(_pfn))->count);           \
   PhysTrack_Add(_vm->physTracker,_pfn);                \
}

#define HOST_UNLOCK_PFN(_vm,_pfn)     {                 \
   ATOMIC_DECREMENT((mem_map+(_pfn))->count);           \
   PhysTrack_Remove(_vm->physTracker,_pfn);             \
}

#define HOST_ISTRACKED_PFN(_vm, _pfn) (PhysTrack_Test(_vm->physTracker, _pfn))


#ifdef SMP_GLOBAL_VMLOCK
#include <asm/spinlock.h>
static spinlock_t vm_lock;
#else
static int vm_lock;
#endif
static int vm_lock_holder;

/*
 *----------------------------------------------------------------------
 *
 * FindMPN --
 *
 *      walks through the hardware defined page tables
 *      to find a valid mpn. 
 *
 * Results:
 *      mpn on success, 0 on failure
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static INLINE MPN
FindMPN(PPN ppn)
{
#if 0
/* def KERNEL_2_3_25 */
	unsigned long addr = ppn << PAGE_SHIFT;
	unsigned long pagenr;
	pgd_t *pgd;
	pmd_t *pmd;
	pte_t *pte;
	
	pgd = pgd_offset(current->mm, addr);
	if (pgd_none(*pgd))
		return 0;
	pmd = pmd_offset(pgd, addr);
	if (pmd_none(*pmd))
		return 0;
	pte = pte_offset(pmd, addr);
	if (!pte_present(*pte))
		return 0;
	pagenr = pte_pagenr(*pte);
	return pagenr;
#else
   int pdoffset = PFN_2_PDOFF(ppn);
   int pgoffset = PFN_2_PGOFF(ppn);
   PTE pte;
   PTE *pteptr;
   PPN ptePPN;
   PDE *directory;
   uint32 cr3reg;
   
   GET_CR3(cr3reg);
   directory = (PDE *) HOST_KERNEL_PA_2_VA(cr3reg); 
   ASSERT(directory);
   ASSERT(directory[pdoffset]);
   ASSERT(directory[pdoffset] & PTE_P);
   
   ptePPN = PTE_2_PFN(directory[pdoffset]);
   pteptr = (PTE *)HOST_KERNEL_PA_2_VA(PPN_2_PA(ptePPN));
   pte = pteptr[pgoffset];
   if (pte & PTE_P) { 
      return PTE_2_PFN(pte);
   } else {
      return 0;
   }
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitSpinLock --
 *
 *      Initialize the vm spin lock
 *
 * Results:
 *      
 *     zero on success, non-zero on error.
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

void
HostIF_InitSpinLock(void)
{
#ifdef SMP_GLOBAL_VMLOCK
   spin_lock_init(&vm_lock);
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_Init --
 *
 *      Initialize the host depandent part of the driver.
 *
 * Results:
 *      
 *     zero on success, non-zero on error.
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

int
HostIF_Init(VMDriver *vm)
{
   vm->memtracker = MemTrack_Init(sizeof(MemTrackEntry));
   vm->physTracker = PhysTrack_Init();
   return (vm->memtracker == NULL || vm->physTracker==NULL);
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsSystemSMP --
 *
 *      define if we are running on a SMP machine
 *
 * Results:
 *      
 *      TRUE if SMP
 *      
 *----------------------------------------------------------------------
 */
Bool    
HostIF_IsSystemSMP(void)
{
#ifdef __SMP__
   return TRUE;
#else
   return FALSE;
#endif
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LookupUserMPN --
 *
 *      Lookup the MPN of a locked user page.
 *
 * Results:
 *      
 *      Returned page is a valid MPN, zero on error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN 
HostIF_LookupUserMPN(VMDriver *vm, char * addr)
{
    MPN mpn;
    mpn = FindMPN(PTR_2_VPN(addr));
    if (mpn==0) {
       NOT_TESTED();
    }
    return mpn;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_InitFP --
 *
 *      masks IRQ13 if not previously the case.
 *
 * Results:
 *      
 *      prevents INTR #0x2d (IRQ 13) from being generated --
 *      assume that Int16 works for interrupt reporting
 *      
 *
 * Side effects:
 *      PIC
 *
 *----------------------------------------------------------------------
 */
void
HostIF_InitFP(VMDriver *vm)
{
   int mask = (1<<(0xd-0x8));
  
   uint8 val = inb(0xA1);
   
#ifdef VMX86_DEVEL
   Log("HostIF_InitFP PIC1=0x%x  %s\n",
       val,
       ((val & mask) ? "Already Masked" : "To mask"));
#endif

   if (!(val & mask)) { 
      val = val | mask;
      outb(val,0xA1);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockPage --
 *
 *      Lockup the MPN of an pinned user-level address space 
 *
 * Results:
 *      
 *      The MPN or zero on an error. 
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
MPN
HostIF_LockPage(VMDriver *vm, void * addr)
{
   MPN mpn;
   VPN vpn;
   MemTrackEntry *entryPtr;
   volatile int c;

   vpn = PTR_2_VPN(addr);
   entryPtr = (MemTrackEntry *) MemTrack_LookupVPN(vm->memtracker, vpn);


   if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
      /*
       * If the page is not in the memory tracker or it has been
       * unlocked. 
       */

      /*
       * get_user forces the page in core
       */
#ifdef KERNEL_2_1
      get_user(c, (char *)addr); 
#else
      c = get_user((char *)addr); 
#endif

      /*
       * now locate it and write the results back to the pmap
       */
      mpn = FindMPN(vpn);
      ASSERT(mpn);
      
      /*
       * XXX SMP.
       * XXX critical region
       */
      if (HOST_ISTRACKED_PFN(vm, mpn)) { 
         Warning("HostIF_LockPage vpn=0x%06x mpn=%06x already tracked\n",vpn,mpn);

         return 0;
      }
      
      HOST_LOCK_PFN(vm,mpn);
      /*
       * If the entry doesn't exist, add it to the memtracker
       * otherwise we just update the mpn.
       */
      if (entryPtr == NULL) {
         entryPtr = MemTrack_Add(vm->memtracker, vpn, mpn);
         if (entryPtr == NULL) {
            HOST_UNLOCK_PFN(vm,mpn);
            return 0;
         }
      } else {
         entryPtr->mpn = mpn;  
      }
   } else {
      /*
       * Already locked. 
       */

      mpn = FindMPN(vpn);
      ASSERT(mpn);
      ASSERT(entryPtr->mpn == mpn);
      ASSERT(HOST_ISTRACKED_PFN(vm, mpn));
   }
   return mpn;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnlockPage --
 *
 *      Unlock an pinned user-level page.
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

int
HostIF_UnlockPage(VMDriver *vm, void * addr)
{
   VPN vpn;
   MemTrackEntry *entryPtr;

   vpn = PTR_2_VPN(addr);
   entryPtr = (MemTrackEntry *) MemTrack_LookupVPN(vm->memtracker, vpn);

   if ((entryPtr == NULL) || (entryPtr->mpn == 0)) {
      Warning("HostIF_UnlockPage vpn=0x%06x not tracked!\n", vpn);
      /* 
       * dangerous, we potentially leak memory here since
       * the use count remains high.
       * This only happens after the watchdog runs and unwedges a 
       * process anyway
       */
      return 1;
   }

   HOST_UNLOCK_PFN(vm,entryPtr->mpn);
   entryPtr->mpn = 0;
   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_FreeAllResources --
 *
 *      Unlock all the pages pinned for a vm. 
 *
 * Results:
 *      0 if successful, otherwise non-zero
 *      
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */

static void 
UnlockEntry(void *clientData,
            MemTrackEntry *entryPtr)
{
   VMDriver *vm = (VMDriver *)clientData;

   if (entryPtr->mpn) {
      if (HOST_ISTRACKED_PFN(vm, entryPtr->mpn)) {
         vm->releaseCount++;
         HOST_UNLOCK_PFN(vm,entryPtr->mpn);
      } else { 
         Warning("UnlockEntry vpn=0x%06x mpn=0x%06x not owned\n",
                 entryPtr->vpn,entryPtr->mpn);
      }
      entryPtr->mpn = 0;
   }
  
}


int
HostIF_FreeAllResources(VMDriver *vm)
{
   vm->releaseCount = 0;
#ifdef KERNEL_2_3_25
   if (vm->crosspagePage) {
      kunmap(vm->crosspagePage);
      vm->crosspagePage = NULL;
   }
#endif
   if (vm->memtracker) {
      MemTrack_Cleanup(vm->memtracker, UnlockEntry,vm);
      vm->memtracker = 0;
   }
   if (vm->physTracker) { 
      PhysTrack_Cleanup(vm->physTracker);
      vm->physTracker = 0;
   }
   
#ifdef notdef_debug
   Warning("HostIF_FreeAllResources pid=%d %d pages released \n",
           (current ? current->pid : -1),
           vm->releaseCount);
#endif
   return 0;
}



/*
 *----------------------------------------------------------------------
 *
 * HostIF_AllocKernelMem
 *
 *      Allocate some kernel memory for the driver. 
 *
 * Results:
 *      The address allocated or NULL on error. 
 *      
 *
 * Side effects:
 *      memory is malloced
 *----------------------------------------------------------------------
 */

void *
HostIF_AllocKernelMem(int size, int wired)
{
   void * ptr = kmalloc(size, GFP_KERNEL);
   
   if (ptr==NULL) { 
      Warning("HostIF_AllocKernelMem failed (size=0%x)\n",size);
   }

   return ptr;
}


void *
HostIF_AllocPage(int wired)
{
   VA vAddr= get_free_page(GFP_KERNEL);
   if (vAddr==0) { 
      Warning("get_free_page() failed\n");
   }
   return (void *)vAddr;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_FreeKernelMem
 *
 *      Free kernel memory allocated for the driver. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      memory is freed.
 *----------------------------------------------------------------------
 */

void
HostIF_FreeKernelMem(void *ptr)
{
   kfree(ptr);
}
   


void
HostIF_FreePage(void *ptr)
{
   VA vAddr = (VA)ptr;
   if (vAddr & (PAGE_SIZE-1)) {
      Warning("HostIF_FreePage 0x%x misaligned\n",ptr);
   } else {
      free_page((VA)ptr);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_RealTime
 *
 *      Read the systems real time clock.
 *
 * Results:
 *      Current real time.
 *
 * Side effects:
 *----------------------------------------------------------------------
 */

VmTimeRealClock
HostIF_ReadTime(void)
{
  struct timeval curtime;
  do_gettimeofday(&curtime);
  return (((VmTimeRealClock)curtime.tv_sec) * 1000000 + curtime.tv_usec);
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_CopyFromUser
 *
 *      Copy memory from the user application into a kernel buffer. 
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

int 
HostIF_CopyFromUser(void *dst, void *src, int len)
{
  int retval;
  retval = verify_area(VERIFY_READ,src, len);
  if (retval) return retval;
  memcpy_fromfs(dst,src,len);
  return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * HostIF_CopytoUser
 *
 *      Copy memory to the user application from a  kernel buffer. 
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

int 
HostIF_CopyToUser(void *dst, void *src, int len)
{
  int retval;
  retval = verify_area(VERIFY_WRITE,dst, len);
  if (retval) return retval;
  memcpy_tofs(dst,src,len);
  return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_UserToDriverPtr 
 *
 *      Convert user address into an address that can be used by the
 *      driver to access the memory. 
 *
 * Results:
 *      A pointer or NULL on error. 
 *
 * Side effects:
 *
 * Bugs:
 *      anyone wants to lock the page?
 *----------------------------------------------------------------------
 */

void *
HostIF_UserToDriverPtr(VMDriver *vm,void *addr)
{
  MPN mpn;
  void *va;
  mpn = FindMPN(PTR_2_VPN(addr));
  
  if (mpn) { 
#ifdef KERNEL_2_3_25
     vm->crosspagePage = mem_map + mpn;
     va = (void*)kmap(vm->crosspagePage);
#else
     /* Cheat in linux since physical address can be accessed from kernel */
     /*  no more cheating in Linux kernel 2.1 ... */
     va = (void *)HOST_KERNEL_PA_2_VA(PPN_2_PA(mpn) + ((uint32)addr & (PAGE_SIZE-1)));
#endif
     
     /*
      * XXX what about locking the page???
      */
     return va;
  } else {
     NOT_TESTED();
     return 0;
  }
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IsMPNLocked
 *
 *      Return entryPtr if MPN is locked into memory
 *
 * Results:
 *      0 if sucessful and non-zero or error.
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static void *
FindFunc(void *arg, MemTrackEntry *entryPtr)
{
   MPN mpn = (MPN) arg;
   
   if (entryPtr->mpn == mpn) {
      return entryPtr;
   }
   return NULL;
}

int
HostIF_IsMPNLocked(VMDriver *vm, MPN mpn)
{
  MemTrackEntry *entry;

  entry = MemTrack_Scan(vm->memtracker,vm, FindFunc);
  return (entry != NULL);
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_CheckMemory
 *
 *      Make sure memory locking is OK.
 *
 *      This function runs as part of the application
 *      process, as such FindMPN, get_user and put_user are fine
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */

static void *
CheckFunc(void *arg, MemTrackEntry *entryPtr)
{
   VMDriver *vm = (VMDriver *)arg;
   MPN mpn;
   
    if (entryPtr->mpn == 0) 
      return NULL;

    mpn = FindMPN(entryPtr->vpn);
    if (mpn==0) { 
       // unmapped
       return entryPtr;
    }

    if (entryPtr->mpn != mpn) {
       Warning("CheckFunc vpn 0x%x mpn 0x%x and not 0x%x \n", 
               entryPtr->vpn, mpn, entryPtr->mpn);
       vm->checkFuncFailed = TRUE;
    }

    
    return NULL;
}



VA
HostIF_CheckMemory(VMDriver *vm)
{
   MemTrackEntry *entryPtr;
   

   vm->checkFuncFailed = FALSE;
   entryPtr = MemTrack_Scan(vm->memtracker,vm, CheckFunc);
   
   if (vm->checkFuncFailed) { 
      return 1;
   } else if (entryPtr) { 
      return (VPN_2_VA(entryPtr->vpn));
   } else {
      return 0;
   }
   
}


#ifdef notdef
/*
 *----------------------------------------------------------------------
 *
 * HostIF_MhzRatingBackup
 *
 *      Return an estimate the of the processor's MHZ rating, based on
 *      the ratio of the cycle counter and the system timer
 *
 * Results:
 *      
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */
#define SPEAKER_PORT	       0x61
#define CLICKS_PER_SEC      1193180
#define COUNTDOWN_FACTOR 7
#define COUNTDOWN ((CLICKS_PER_SEC+(1<<(COUNTDOWN_FACTOR-1)))/(1<<COUNTDOWN_FACTOR))

uint32
HostIF_MhzRatingBackup(VMDriver *vm)
{
   uint8 byte;
   VmTimeRealClock startTSC, endTSC;
   uint32 flags, count = 0;

   count = 0;
   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();

   /* set up Timer 2 to count down */
   byte = inb(SPEAKER_PORT);
   byte = (byte & ~0x2) | 0x1;
   outb(byte, SPEAKER_PORT);
   outb(0xb0, 0x43);
   outb_p((COUNTDOWN & 0xff), 0x42);
   outb((COUNTDOWN >> 8), 0x42);	

   /* count number of processor cycles for Timer 2 countdown */
   startTSC = GET_TSC();
   while ((inb(SPEAKER_PORT) & 0x20) == 0) {
      count++;
   }
   endTSC = GET_TSC();
   endTSC -= startTSC;

   RESTORE_FLAGS(flags);

   if ((count == 0) || (endTSC > 0xffffffff) || (endTSC < COUNTDOWN)) {
      /* detected faulty timer, or cycle counter */
      return 0;
   } else {
      return (((uint32) endTSC << COUNTDOWN_FACTOR)/1000000);
   }
}
#endif

/*
 *----------------------------------------------------------------------
 *
 * HostIF_MhzRating
 *
 *      Return an estimate the of the processor's MHZ rating, based on
 *      the ratio of the cycle counter and gettimeofday
 *
 * Results:
 *      
 *
 * Side effects:
 *      None.
 *----------------------------------------------------------------------
 */
uint32
HostIF_MhzRating(VMDriver *vm)
{

   VmTimeType vDiff, rDiff;
   uint32 flags;

   /*
    * Compute the cycle rate based on the change in the
    * cycle counter and real time clock since we open the 
    * device.
    */

   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();
   vDiff = GET_TSC() - vm->startTime.count;
   rDiff = HostIF_ReadTime() - vm->startTime.time;
   RESTORE_FLAGS(flags);

   if (rDiff == 0) { 
      return 0;  /* Failure */
   } 
   /* Need to do 32bit divide since 64bit one doesn't seem to 
    * work in a linux device driver. Scale the numbers back into
    * 32bit numbers. 
    */
   while (vDiff > (0xffffffff/10)) {
      vDiff >>= 1;
      rDiff >>= 1;
   }
   return ((10*(uint32)vDiff)/(uint32)rDiff + 5)/10;
}
   

/*
 *----------------------------------------------------------------------
 *
 * HostIF_LockKernel --
 *
 *      grabs the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock held
 *
 *----------------------------------------------------------------------
 */
void
HostIF_LockKernel(void)
{
#ifdef __SMP__
   lock_kernel();
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_UnLockKernel --
 *
 *      releases the global kernel lock
 *      For the UP driver, do nothing
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      lock released
 *
 *----------------------------------------------------------------------
 */
void
HostIF_UnLockKernel(void)
{
#ifdef __SMP__
   unlock_kernel();
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_GlobalVMLock --
 *
 *      grabs the global data structure lock.
 *      For the UP driver, assert that the lock counter is zero and increment
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      Should be a very low contention lock.
 *
 *----------------------------------------------------------------------
 */
void
HostIF_GlobalVMLock(int callerID)
{
#ifdef SMP_GLOBAL_VMLOCK
   if (! spin_trylock(&vm_lock)) {
      printk(KERN_WARNING "/dev/vmmon: %d wait for global VM lock %d\n",
	     callerID, vm_lock_holder);
      spin_lock(&vm_lock);
   }
#else
   if (vm_lock) { 
      Panic("/dev/vmmon: 0x%x global VM lock held -- no SMP version\n");
   }
   vm_lock++;
#endif
   vm_lock_holder = callerID;
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_GlobalVMUnlock --
 *
 *      releases the global data structure lock.
 *      For the UP driver, assert that the lock counter is 1 and decrement
 *
 * Results:
 *      
 *      void
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
HostIF_GlobalVMUnLock(int callerID)
{
   if (vm_lock_holder != callerID) {
      printk(KERN_WARNING "/dev/vmmon: %d releasing global VM lock %d\n",
	     callerID, vm_lock_holder);
   }
   vm_lock_holder = -1;
#ifdef SMP_GLOBAL_VMLOCK
   spin_unlock(&vm_lock);
#else
   if (vm_lock != 1) { 
      Panic("/dev/vmmon: 0x%x global lock not held -- no SMP version\n");
   }
   vm_lock--;
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_APICBase --
 *
 *      Lookup the APIC physical address
 *
 * Results:
 *      
 *      Returned value is APIC physical address
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
/*
 * Utility routines 
 */
#if defined(KERNEL_2_1) && defined(X86_APIC_PRESENT)
#ifdef PROBE_APIC
static uint64
GetMSR(int index)
{
   uint64 msr;
   msr = __GET_MSR (index);
   return msr;
}

static void
GetCPUID(int index, uint32 *regs)
{
   __GET_CPUID (index, regs);
}
#endif
#endif

MA
HostIF_APIC_Base(VMDriver *vm, Bool setVMPtr)
{
#if defined(KERNEL_2_1) && defined(X86_APIC_PRESENT)

   void *va;
   MPN mpn;

#ifdef PROBE_APIC
   uint32 regs[4];
   uint32 *ptr;
   char name[16];
   uint8 family;
   uint64 msr;

   GetCPUID(0, regs);
   ptr = (uint32 *)name;
   ptr[0] = regs[1];
   ptr[1] = regs[3];
   ptr[2] = regs[2];
   ptr[3] = 0;

   if (strcmp(name,"GenuineIntel") == 0) {
      GetCPUID(1, regs);
      if ((regs[3] & CPUID_FEATURE_MSR) &&
	  (regs[3] & CPUID_FEATURE_APIC)) {
	 /* APIC is present and enabled */
	 family = (regs[0] >> 8) & 0xF;
	 if (family == 6) {
	    /* P6 allows APIC relocation, encoded in MSR 1B */
	    msr = GetMSR(0x1b);
	    mpn = (msr >> 12) & 0xFFFFFF;
	    return MPN_2_MA(mpn);
	 } else {
	    return APIC_DEFAULT_ADDRESS;
	 }
      }
   }
#endif   

   va = (void *)__fix_to_virt(FIX_APIC_BASE);
   if (setVMPtr) {
      vm->hostAPIC = va;
   } else {
      vm->hostAPIC = NULL;
   }
   mpn = FindMPN(PTR_2_VPN(va));
   return MPN_2_MA(mpn);

#else

   return 0;

#endif
}


/*
 *----------------------------------------------------------------------
 *
 * HostIF_IOAPICBase --
 *
 *      Lookup the IO-APIC physical address
 *
 * Results:
 *      
 *      Returned value is IO-APIC physical address
 *      
 *
 * Side effects:
 *     None
 *
 *----------------------------------------------------------------------
 */
#define PCI_CONF_ADDR_REG_PORT   0xcf8
#define PCI_CONF_DATA_REG_PORT   0xcfc
#define VMWARE__PCI_DEVICE_ID_INTEL_82371SB 0x70008086
#define VMWARE__PCI_DEVICE_ID_INTEL_82371AB 0x71108086

MA
HostIF_IOAPIC_Base(VMDriver *vm)
{
#if defined(KERNEL_2_1) && defined(X86_IOAPIC_PRESENT)
   int bus, slot, confword;
   uint32 id, apicbase;
   MPN mpn;
   char *addr;

   bus = 0;
   for (slot = 0; slot < 32; slot++) {
      confword = 0x80000000 | (bus << 16) |  (slot << 11);
      OUT32(PCI_CONF_ADDR_REG_PORT, confword);
      id = IN32(PCI_CONF_DATA_REG_PORT);
      if ((id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ||
	  (id == VMWARE__PCI_DEVICE_ID_INTEL_82371AB)) {
#ifdef VMX86_DEVEL
	 Log("IOAPIC: detected %s on PCI bus %d, slot %d\n",
	     (id == VMWARE__PCI_DEVICE_ID_INTEL_82371SB) ? "PIIX3" : "PIIX4",
	     bus, slot);
#endif
	 confword |= 20; /* register 80 */
	 OUT32(PCI_CONF_ADDR_REG_PORT, confword);
	 apicbase = IN32(PCI_CONF_DATA_REG_PORT);
	 apicbase = apicbase & 0x3f;
	 return (IOAPIC_DEFAULT_ADDRESS | (apicbase << 10));
      }
   }
   
   addr = (char *)__fix_to_virt(VMWARE__FIX_IO_APIC_BASE);
   mpn = FindMPN(PTR_2_VPN(addr));
   return MPN_2_MA(mpn);
#else
   return 0;
#endif
}


