/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * **********************************************************/
#ifdef VMX86_DEVEL
char rcsId_driver[] = "$Id: driver.c,v 1.7.2.1 2000/06/09 00:21:48 hpreg Exp $";
#else
#define FILECODE "F(300)"
#endif 

#include "driver-config.h"

#ifdef KERNEL_2_1
#define EXPORT_SYMTAB
#endif

extern int errno;
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/malloc.h>

#ifdef __SMP__
#include <linux/smp.h>
#include <linux/smp_lock.h>
#endif

#ifdef KERNEL_2_1
#include <linux/poll.h>
#include <linux/interrupt.h>
#endif

#ifdef DO_APM
#include <linux/apm_bios.h>
static int LinuxDriverAPMCallback(apm_event_t event);
static int LinuxDriverAPMstate = APM_STATE_READY;
#endif

#define __KERNEL_SYSCALLS__
#include <asm/unistd.h>
#include <asm/io.h>

#include "x86.h"
#include "vm_types.h"
#include "vm_assert.h"
#include "modulecall.h"
#include "vm_asm.h"
#include "vm_time.h"
#include "vmx86.h"
#include "initblock.h"
#include "task.h"
#include "hostif.h"
#include "driver.h"
#include "speaker_reg.h"
#include "vtrace.h"
#include "memtrack.h"

#ifdef VMX86_DEVEL
#include "private.h"
#endif

#ifdef SUPPORT_PASSTHROUGH
#include "passthrough.h"	// _driver_ version of passthrough.h 
#endif

int errno;       // required for _exit()
static void LinuxDriverQueue(VMLinux *vmLinux);
static void LinuxDriverDequeue(VMLinux *vmLinux);
static Bool LinuxDriverCheckPadding(void);


#ifdef VMX86_DEVEL
#define TAGNAME "(development)"
#else
#define TAGNAME "$Name: build-621 $"
#endif

/*
 * the product of these two numbers is the time until a deadlock
 * is detected and cleared. Currently set to 20 seconds
 */

#define ZOMBIE_CHECK_INTERVAL 200 /* ticks, 2 seconds */
#define ZOMBIE_BLOCKED_COUNT   10  /* should be 10 or 20 */

struct vmx_linux_state linuxState;


/*
 *----------------------------------------------------------------------
 *
 * Device Driver Interface --
 *
 *      Runs the VM by implementing open/close/ioctl functions
 *
 *
 *----------------------------------------------------------------------
 */
static int LinuxDriver_Open(struct inode * inode, struct file * filp);

static int LinuxDriver_Ioctl( struct inode *inode, struct file *filp,
         u_int iocmd, unsigned long ioarg );


#ifdef KERNEL_2_1
static int LinuxDriver_Close( struct inode *inode, struct file *filp );
static unsigned int LinuxDriver_Poll(struct file *file, poll_table *wait);
#else
static void LinuxDriver_Close( struct inode *inode, struct file *filp );
static int LinuxDriver_Select(struct inode *inode, struct file *filp,
			      int sel_type, select_table *wait);
#endif

static void LinuxDriverSelectTimeout(VMLinux *vmLinux);

#ifdef KERNEL_2_1
static struct file_operations vmuser_fops = {
   NULL,      /* lseek */
   NULL,      /* read */
   NULL,      /* write */
   NULL,      /* readdir */
   LinuxDriver_Poll,	/* select */
   LinuxDriver_Ioctl,   /* ioctl */
   NULL,      /* mmap */
   LinuxDriver_Open,   /* just a selector for the real open */
   NULL,		/* flush */
   LinuxDriver_Close,   /* release */
   NULL      /* fsync */
};
#else
static struct file_operations vmuser_fops = {
   NULL,      /* lseek */
   NULL,      /* read */
   NULL,      /* write */
   NULL,      /* readdir */
   LinuxDriver_Select,	/* select */
   LinuxDriver_Ioctl,   /* ioctl */
   NULL,      /* mmap */
   LinuxDriver_Open,   /* just a selector for the real open */
   LinuxDriver_Close,   /* release */
   NULL      /* fsync */
};
#endif



/*
 *----------------------------------------------------------------------
 *
 * VMX86_RegisterMonitor --
 *
 *      (debugging support) Should be the first function of this file
 *
 * Results:
 *      
 *      Registers the module. 
 *      /sbin/ksyms -a | grep VMX86_RegisterMonitor will return the base
 *      address of that function as loaded in the kernel. 
 * 
 *      Since this is the first function of the kernel module,
 *      every other symbol can be computing by adding the base
 *      to the output of nm.
*
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
int VMX86_RegisterMonitor(int);

#ifdef KERNEL_2_1
EXPORT_SYMBOL(VMX86_RegisterMonitor);
#else
static struct symbol_table vmx86_syms = {
#include <linux/symtab_begin.h>
   X(VMX86_RegisterMonitor),
#include <linux/symtab_end.h>
};
#endif

int 
VMX86_RegisterMonitor(int value)
{
   printk("/dev/vmmon: RegisterMonitor(%d) \n",value);
   return 1291;
}

/*
 *----------------------------------------------------------------------
 *
 * init_module --
 *
 *      linux module entry point. Called by /sbin/insmod command
 * 
 * Results: 
 *      registers a device driver for a major # that depends
 *      on the uid. Add yourself to that list.  List is now in
 *      private/driver-private.c.
 *
 *----------------------------------------------------------------------
 */
int 
init_module(void)
{
   int retval;

   if (!LinuxDriverCheckPadding()) { 
      return -ENOEXEC;
   }
   
   HostIF_InitSpinLock();

#ifdef VMX86_DEVEL
   devel_init_module();
   linuxState.minor = 0;
   retval = register_chrdev( linuxState.major, linuxState.deviceBuf, &vmuser_fops );
#else
   sprintf(linuxState.deviceBuf,"vmmon");
   linuxState.major = 10;
   linuxState.minor = 165;
   linuxState.misc.minor = linuxState.minor;
   linuxState.misc.name = linuxState.deviceBuf;
   linuxState.misc.prev = 0;
   linuxState.misc.next = 0;
   linuxState.misc.fops = &vmuser_fops;
   
   retval = misc_register(&linuxState.misc);
#endif
   
   if (retval) {
      Warning("Module %s: error registering with major=%d minor=%d tag=%s\n", linuxState.deviceBuf,linuxState.major, linuxState.minor, TAGNAME);
      return -ENOENT;
   }
   Log("Module %s: registered with major=%d minor=%d tag=%s\n", linuxState.deviceBuf, linuxState.major, linuxState.minor, TAGNAME);
   
#ifndef KERNEL_2_1      
   retval = register_symtab(&vmx86_syms);
#endif
   Log("Module %s: initialized\n", linuxState.deviceBuf);

   return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * cleanup_module --
 *
 *      Called by /sbin/rmmod
 *
 *
 *----------------------------------------------------------------------
 */

int cleanup_module(void)
{
   int retval;
   
   /*
    * XXX smp race?
    */
#ifdef VMX86_DEVEL 
   retval =  unregister_chrdev( linuxState.major, linuxState.deviceBuf );
#else
   retval = misc_deregister(&linuxState.misc);
#endif

   if (retval) {
      Warning("Module %s: error unregistering\n", linuxState.deviceBuf);
   } else {
      Log("Module %s: unloaded\n", linuxState.deviceBuf);
   }
   return 0;
}



/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Open  --
 *
 *      called on open of /dev/vmmon or /dev/vmx86.$USER. Use count used
 *      to determine eventual deallocation of the module 
 *
 * Side effects:
 *     Increment use count used to determine eventual deallocation of 
 *     the module 
 *
 *----------------------------------------------------------------------
 */
static int 
LinuxDriver_Open(struct inode * inode, 
                 struct file * filp)
{
   VMLinux *vmLinux;
   VMDriver *vm;
   uint32 flags;

   if ((filp->f_flags & O_ACCMODE) == O_WRONLY) {
#ifdef VMX86_DEVEL
      return devel_suid();
#else
      return -EPERM;
#endif
   }

   vmLinux = (VMLinux *) kmalloc(sizeof (VMLinux), GFP_KERNEL);
   if (vmLinux == NULL) {
      Warning("LinuxDriver_Open kmalloc failed\n");
      return -ENOMEM;
   }
   memset(vmLinux, 0, sizeof *vmLinux);
#ifdef KERNEL_2_3_1
   init_waitqueue_head(&vmLinux->selectQueue);
#endif

   
   vm = Vmx86_Init((void *)filp, (void *)(current->pid));
   if (vm == NULL) {
      Warning("Vmx86_Init failed\n");
      kfree(vmLinux);
      return -ENOMEM;
   }
   
   
      
   /*
    * Snap shot the time stamp counter and the real time so we
    * can later compute an estimate of the cycle time.
    */
   SAVE_FLAGS(flags); 
   CLEAR_INTERRUPTS();
   vm->startTime.time = HostIF_ReadTime(); 
   vm->startTime.count = GET_TSC();
   RESTORE_FLAGS(flags);

   vmLinux->vm = vm;
   vmLinux->selectTimer.data = (unsigned long) vmLinux;
   vmLinux->selectTimer.function =
      (void (*)(unsigned long)) LinuxDriverSelectTimeout;
   filp->private_data = vmLinux;
  
   
   
   LinuxDriverQueue(vmLinux);
   
#ifdef SUPPORT_LINUXVMWARE
   VMWare_SDumpInit();
   VMWare_SDumpSend("Real men use serial ports! \n");
#endif
   
   //printk("/dev/vmmon: useCount INC pid=%d \n",current->pid);
   MOD_INC_USE_COUNT;

#ifdef DO_APM
   apm_register_callback(&LinuxDriverAPMCallback);
#endif
   
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Close  --
 *
 *      called on close of /dev/vmmon or /dev/vmx86.$USER, most often when the 
 *      process exits. Decrement use count, allowing for possible uninstalling
 *      of the module.
 *
 *----------------------------------------------------------------------
 */

#ifdef KERNEL_2_1
static int
#else
static void 
#endif
LinuxDriver_Close(struct inode * inode, 
                  struct file * filp)
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;

#ifdef SUPPORT_LINUXVMWARE
   VMWare_SetVTracer(0);
#endif

   if (!vmLinux->vm) { 
      printk("/dev/vmmon: (zombie) close fd for pid=%d\n",
             current->pid);
   } else {
      //printk("/dev/vmmon: useCount DEC pid=%d\n",vmLinux->pids[0]);
   }
   
   MOD_DEC_USE_COUNT;



   LinuxDriverDequeue(vmLinux);
   
   if (vmLinux->vm != NULL) { 
      Vmx86_DestroyVM(vmLinux->vm);
   }

   /* Just in case something happens and the release ioctl is never made */
#ifdef SUPPORT_PASSTHROUGH
   Passthrough_Release(vmLinux);
#endif

   

   kfree(vmLinux);
   filp->private_data = NULL;
#ifdef KERNEL_2_1
   return 0;
#endif
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Select  --
 *
 *      This is only used to wake up the idle select() at the next
 *	clock tick.
 *	We don't care what kind of select this is (read, write, or exception).
 *
 *----------------------------------------------------------------------
 */
#ifdef KERNEL_2_1

unsigned int 
LinuxDriver_Poll(struct file *filp,
		 poll_table *wait)
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   unsigned long flags;

   save_flags(flags);
   cli();
   if (wait != NULL) {
      HostIF_GlobalVMLock(11);	// protect access to vmLinux
      if (vmLinux->selectWaiting) {
	 del_timer(&vmLinux->selectTimer);
      } else {
	 vmLinux->selectWaiting = TRUE;
      }
      poll_wait(filp, &vmLinux->selectQueue, wait);
      init_timer(&vmLinux->selectTimer);
      vmLinux->selectTimer.expires = jiffies + 1;
      add_timer(&vmLinux->selectTimer);
      HostIF_GlobalVMUnLock(11);
   }
   restore_flags(flags);

   return !vmLinux->selectWaiting;
}

#else		   

int
LinuxDriver_Select(struct inode *inode,
                   struct file *filp,
                   int sel_type,
                   select_table *wait)
{
#if 0
   if (current->timeout) {
      current->timeout = jiffies + 1;
   }
   return 0;
#else
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   unsigned long flags;

   save_flags(flags);
   cli();
   if (wait != NULL) {
      if (vmLinux->selectWaiting) {
	 del_timer(&vmLinux->selectTimer);
      } else {
	 vmLinux->selectWaiting = TRUE;
      }
      select_wait(&vmLinux->selectQueue, wait);
      vmLinux->selectTimer.next = vmLinux->selectTimer.prev = NULL;
      vmLinux->selectTimer.expires = jiffies + 1;
      add_timer(&vmLinux->selectTimer);
   }
   restore_flags(flags);

   return !vmLinux->selectWaiting;
#endif
}

#endif
/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverSelectTimeout  --
 *
 *      Wake up a process waiting in select.
 *
 *----------------------------------------------------------------------
 */

void
LinuxDriverSelectTimeout(VMLinux *vmLinux)
{
   // We are already cli()
   vmLinux->selectWaiting = FALSE;
   wake_up(&vmLinux->selectQueue);
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverIPIHandler  --
 *
 *      Null IPI handler - for monitor to notice AIO completion
 *
 *----------------------------------------------------------------------
 */
void
LinuxDriverIPIHandler(void *info)
{
   return;
}


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriver_Ioctl --
 *
 *      Main path for UserRPC
 *
 * Results:
 *      
 *       
 *      
 *      
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static int 
LinuxDriver_Ioctl( struct inode *inode, 
                   struct file *filp,
                   u_int iocmd, 
                   unsigned long ioarg )
{
   VMLinux *vmLinux = (VMLinux *) filp->private_data;
   VMDriver *vm = vmLinux->vm;
   int retval = 0;
   int err;
   InitBlock initParams;


#ifdef SUPPORT_LINUXVMWARE
   VMWare_SetVTracer(VTrace_Set);
#endif

   if (!vm) {
      /*
       * Allow programs to obtain VM memory info from the driver
       */
      if (iocmd != IOCTLCMD_GET_MEM_INFO) {
         return -EDEADLK;
      }
   }


   switch (iocmd) {
   case IOCTLCMD_INIT:
      ASSERT(vm != NULL);
      err = HostIF_CopyFromUser(&initParams,(char*)ioarg,sizeof(InitBlock));
      ASSERT(!err);
      err = Vmx86_CreateVM(vm, &initParams);
      if (err) { 
	 retval = -1;
      }
      err = HostIF_CopyToUser((char*)ioarg,&initParams,sizeof(InitBlock));
      ASSERT(!err);
      
      break;

   case IOCTLCMD_LATE_INIT:
      ASSERT(vm != NULL);
      Vmx86_LateInit(vm);
      break;
      
   case IOCTLCMD_RUN:
      ASSERT(vm != NULL);
#ifdef __SMP__
      unlock_kernel();
#endif
      retval = Vmx86_RunVM(vm);
#ifdef __SMP__
      lock_kernel();
#endif
      break;

   case IOCTLCMD_BEEP: { // XXX for buggy Linux
      uint8 byte;
      byte = inb(SPEAKER_PORT);
      if (ioarg >> 16) {
	 byte |= (SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      } else {
	 byte &= ~(SPEAKER_ENABLE_SPEAKER | SPEAKER_TIMER2_GATE);
      }
      outb(byte, SPEAKER_PORT);
      outb(0xb6, 0x43);
      outb_p((uint8) ioarg, 0x42);
      outb((uint8) (ioarg >> 8), 0x42);
      break;
   }

   // XXX Deprecated.  Now handled by devel_suid() for certain types of
   // open() calls.
   case IOCTLCMD_SETUID:
      retval = -1;
      break;

   case IOCTLCMD_LOOKUPMPN: { 
     char *addr = (char *)ioarg;
     MPN mpn;
     mpn = HostIF_LookupUserMPN(vm,addr);
     retval = mpn;
     break;
   }
   
   case IOCTLCMD_LOCKPAGE: {
      char *addr = (char *)ioarg;
      MPN mpn;
      mpn = Vmx86_LockPage(vm, addr, TRUE);
      retval = mpn;
      break;
   }
   
   case IOCTLCMD_UNLOCKPAGE: { 
      char *addr = (char *)ioarg;
      retval = Vmx86_UnlockPage(vm, addr, TRUE);
      break;
   }

   case IOCTLCMD_GET_NUM_VMS : {
      retval = Vmx86_GetNumVMs();
      break;
   }
      
   case IOCTLCMD_SET_HARD_LIMIT: {
      int32 limit;
      err = HostIF_CopyFromUser(&limit, (void *)ioarg, sizeof(limit));
      if (err) {
         retval = -EINVAL;
      } else if (!Vmx86_SetLockedPagesLimit(limit)) {
         retval = -EINVAL;
      } else {
         retval = 0;
      }
      break;
   }

   case IOCTLCMD_GET_HARD_LIMIT: {
      retval = Vmx86_GetLockedPagesLimit();
      break;
   }

   case IOCTLCMD_SET_MEM_INFO: {
      VMGetSetMemInfoInArgs inArgs;

      err = HostIF_CopyFromUser(&inArgs, (void *)ioarg, sizeof(inArgs));
      if (err) {
         retval = -EINVAL;
      } else {
         Vmx86_SetMemInfo(vm, &inArgs);
         retval = 0;
      }
      break;
   }

   case IOCTLCMD_GET_MEM_INFO: {
      if (!Vmx86_GetMemInfoCopy(vm, (VMGetSetMemInfoOutArgs *)ioarg)) {
         retval = -EINVAL;
      } else {
         retval = 0;
      }
      break;
   }

   case IOCTLCMD_GET_VM_LIST: {
      if (!Vmx86_GetVMListCopy((VMGetVMListResult *)ioarg)) {
         retval = -EINVAL;
      } else {
         retval = 0;
      }
      break;
   }
   case IOCTLCMD_GET_VM_INFO: {
      VMGetVMInfoResult info;

      err = HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info));
      if (err) {
         retval = -EINVAL;
      } else if (!Vmx86_GetVMInfo(info.vmUID, &info)) {
         retval = -EINVAL;
      } else {
         err = HostIF_CopyToUser((void *)ioarg, &info, sizeof(info));
         if (err) {
            retval = -EINVAL;
         } else {
            retval = 0;
         }
      }
      break;
   }
   case IOCTLCMD_SET_VM_INFO: {
      VMSetVMInfoArgs info;

      err = HostIF_CopyFromUser(&info, (void *)ioarg, sizeof(info));
      if (err) {
         retval = -EINVAL;
      }
      
      Vmx86_SetVMInfo(vm, &info);
      retval = 0;
      break;
   }

   case IOCTLCMD_GETSTATS: {
      err = HostIF_CopyToUser((void *)ioarg, &vm->stats, sizeof(vm->stats));
      if (err) {
         retval = -EINVAL;
      }
      break;
   }

   case IOCTLCMD_SETSTATS: {
      err = HostIF_CopyFromUser(&vm->stats, (void *)ioarg, sizeof(vm->stats));
      if (err) {
         retval = -EINVAL;
      }
      break;
   }
 
   case IOCTLCMD_DECLARE_SLAVE :
      /* obsolete */
      retval = 0;
      break;
      
   case IOCTLCMD_ISMPSAFE : { 
#ifdef __SMP__
      retval = TRUE;
#else
      retval = FALSE;
#endif
      break;
   }
   
   case IOCTLCMD_APICBASE: { 
      MA ma;
      Bool setVMPtr = (Bool) ioarg;
#if defined(LINUX_VERSION_CODE) && defined(KERNEL_VERSION)
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,20)
      // Kernel uses NMIs for deadlock detection -
      //  set APIC VMptr so that NMIs get disabled in the monitor
      setVMPtr = TRUE;
#endif
#endif
      ma = HostIF_APIC_Base(vm, setVMPtr);
      retval = ma;
      break;
   }
   
   case IOCTLCMD_IOAPICBASE: { 
      MA ma;
      ma = HostIF_IOAPIC_Base(vm);
      retval = ma;
      break;
   }

   
   case IOCTLCMD_CHECK_MEMORY : 
      retval = HostIF_CheckMemory(vm);
      break;
      
   case IOCTLCMD_REGISTER_PASSTHROUGH_IRQ: {
#ifdef SUPPORT_PASSTHROUGH
      retval = Passthrough_RegisterIRQ((unsigned char) ioarg, "VMware", vmLinux);
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_REGISTER_PASSTHROUGH_IO: {
#ifdef SUPPORT_PASSTHROUGH
      struct passthrough_iorange ior;

      err = HostIF_CopyFromUser(&ior, (char*)ioarg, sizeof(ior));
      retval = Passthrough_RegisterIORegion(ior.ioBase, ior.numPorts, "VMware");
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_FREE_PASSTHROUGH_IRQ: {
#ifdef SUPPORT_PASSTHROUGH
      Passthrough_FreeIRQ((unsigned char) ioarg, vmLinux);
      retval = 0;
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_FREE_PASSTHROUGH_IO: {
#ifdef SUPPORT_PASSTHROUGH
      struct passthrough_iorange ior;

      err = HostIF_CopyFromUser(&ior, (char*)ioarg, sizeof(ior));
      retval = Passthrough_ReleaseIORegion(ior.ioBase, ior.numPorts);
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_START_PASSTHROUGH: {
#ifdef SUPPORT_PASSTHROUGH
      retval = Passthrough_Init(vmLinux);
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_STOP_PASSTHROUGH: {
#ifdef SUPPORT_PASSTHROUGH
      retval = Passthrough_Release(vmLinux);
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_QUERY_PASSTHROUGH: {
#ifdef SUPPORT_PASSTHROUGH
      unsigned char irq;
      if (vmLinux->numPendingPassthroughIRQs > 0) {
        irq = vmLinux->pendingPassthroughIRQs[--vmLinux->numPendingPassthroughIRQs],
        retval = vmLinux->numPendingPassthroughIRQs;
      } else {
        irq = 0x0;
        retval = -1;
      }
      err = HostIF_CopyToUser((unsigned char*)ioarg, &irq, sizeof(unsigned char));
      ASSERT(!err);
#else
      retval = -EINVAL;
#endif
      break;
   }

   case IOCTLCMD_ALLOW_CORE_DUMP:
      if (current->euid == current->uid &&
	  current->fsuid == current->uid &&
          current->egid == current->gid &&
	  current->fsgid == current->gid) {
	 current->dumpable = 1;
      }
      break;

   case IOCTLCMD_BROADCAST_IPI:
#ifdef __SMP__
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 2, 8)
      retval = smp_call_function(LinuxDriverIPIHandler, NULL, FALSE, FALSE);
#endif
#endif
      break;

   case IOCTLCMD_FREE_RESOURCES:
      retval = HostIF_FreeAllResources(vm);
      break;

   default:
      Warning("Unknown ioctl(0x%x)\n", iocmd);
      retval = -EINVAL;
   }


#ifdef SUPPORT_LINUXVMWARE
   VMWare_SetVTracer(0);
#endif

   return retval;
}




/*
 *----------------------------------------------------------------------
 *
 * vLog --
 *
 *     Log() from the kernel module logged in current->files[fd]
 *
 *
 *----------------------------------------------------------------------
 */   
static INLINE 
void vLog(int fd)
{
   if (fd < 0) {
      return;
   }
   ASSERT(strlen(linuxState.buf));
   ASSERT(strlen(linuxState.buf) < LINUXLOG_BUFFER_SIZE);

   if (current->files && 
       current->files->fd[fd] &&
       current->files->fd[fd]->f_op) {
      struct file *file = current->files->fd[fd];

#ifdef KERNEL_2_1
      mm_segment_t old_fs = get_fs();
      set_fs(get_ds());    
      file->f_op->write(file, (char*)linuxState.buf, strlen(linuxState.buf), &file->f_pos);
#else
      int32 old_fs = get_fs();
      set_fs(get_ds());    
      file->f_op->write(file->f_inode, file, (char*)linuxState.buf, strlen(linuxState.buf));
#endif
      set_fs(old_fs);

   } 
}
   
/*
 *----------------------------------------------------------------------
 *
 * vWarning --
 *
 *      Warning() get here. Adds a \r to compensate for the 
 *      uncooked mode of the terminal.
 *
 *----------------------------------------------------------------------
 */
static INLINE 
void vWarning(VMDriver *vm)
{
   int fd = 1;
   
   ASSERT(strlen(linuxState.buf));
   ASSERT(strlen(linuxState.buf) < LINUXLOG_BUFFER_SIZE);

   if (vm && 
       current->files && 
       current->files->fd[fd] &&
       current->files->fd[fd]->f_op) {
      struct file *file = current->files->fd[fd];
      
#ifdef KERNEL_2_1
      mm_segment_t old_fs = get_fs();
      set_fs(get_ds());    
      file->f_op->write(file, (char*)linuxState.buf, strlen(linuxState.buf), &file->f_pos);
#else
      int32 old_fs = get_fs();
      set_fs(get_ds());    
      file->f_op->write(file->f_inode, file, (char*)linuxState.buf, strlen(linuxState.buf));
#endif
      
      set_fs(old_fs);
      
      printk("/dev/vmmon: (pid=%d) %s",current->pid,linuxState.buf);
      
   } else {
      /* Use the kernel log */
      printk(KERN_WARNING "/dev/vmmon: %s", linuxState.buf);
   }
}



/*
 *----------------------------------------------------------------------
 *
 * Warning --
 *
 *      Warning messages from kernel module: logged to stdout and the log file
 *
 *----------------------------------------------------------------------
 */
void 
Warning(char *fmt,...)
{
   VMDriver *vm;
   va_list args;
   

   vm = Vmx86_GetVMforProcess((void *)(current->pid));

   va_start(args, fmt);
   vsprintf(linuxState.buf, fmt, args); 
   va_end(args);
   
   if (vm) { 
      vLog(vm->logFD);
   }
   vWarning(vm);
}

/*
 *----------------------------------------------------------------------
 *
 * Log --
 *
 *      Log messages from kernel module: logged to log file only
 *
 *----------------------------------------------------------------------
 */
void 
Log(char *fmt,...)
{
   VMDriver *vm;
   va_list args;


   vm = Vmx86_GetVMforProcess((void *)(current->pid));
  
   va_start(args, fmt);
   vsprintf(linuxState.buf, fmt, args); 
   va_end(args);
   
   if (vm) { 
      vLog(vm->logFD);
   } else {
      /* Use the kernel log with at least a KERN_DEBUG level so that it doesn't
         garbage the screen at (re)boot time on RedHat 6.0 --hpreg */
      printk(KERN_DEBUG "/dev/vmmon: %s", linuxState.buf);
   }
}


/*
 *----------------------------------------------------------------------
 *
 * Panic --
 *
 *      ASSERTION failures and Panics from kernel module get here.
 *      Message is logged to stdout and the log file      
 *      
 *
 * Side effects:
 *      Never returns
 *
 *----------------------------------------------------------------------
 */
void
Panic(char *fmt, ...)
{
   asmlinkage int sys_exit(int exit_code);
#ifndef KERNEL_2_1
   extern int intr_count;
#endif
   VMDriver *vm = Vmx86_GetVMforProcess((void *)(current->pid));
   va_list args;

   va_start(args, fmt);
   vsprintf(linuxState.buf, fmt, args); 
   va_end(args);

   /*
    * XXX 
    * XXX We cannot exit() the process since we are not running it
    * XXX
    */
#ifdef KERNEL_2_1
   if (in_interrupt()) {
      printk("/dev/vmmon: Panic in interrupt (no intrcount)\n");
      panic("Assertion failure in interrupt handling in VMX86\n");
   }
   
#else
   if (intr_count) {
      printk("/dev/vmmon: Panic intrcount=%d\n",intr_count);
      panic("Assertion failure in interrupt handling in VMX86\n");
   }
#endif
   
   if (vm) { 
     vLog(vm->logFD);
     vWarning(vm);
     sprintf(linuxState.buf,"VMX86 driver panic. pid=%d\n\r",current->pid);  
     vLog(vm->logFD);
     vWarning(vm);
   }
   
   _exit(1);
}



/*
 *----------------------------------------------------------------------
 *
 * LinuxDriverQueue --
 *
 *      add the vmLinux to the global queue
 *
 * Results:
 *      
 *      void
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */
static void
LinuxDriverQueue(VMLinux *vmLinux)
{
   Bool installWatchdog;
   /*
    * insert in global vm queue
    */
   
   HostIF_GlobalVMLock(12);
   
   installWatchdog = (linuxState.head == NULL);
   vmLinux->next = linuxState.head;
   linuxState.head = vmLinux;
   

   HostIF_GlobalVMUnLock(12);	
   
} 


/*
 *----------------------------------------------------------------------
 *
 * LinuxDriveDequeue --
 *
 *      remove from active list
 *
 * Results:
 *      
 *      void
 * Side effects:
 *      printk if it is not in the list (error condition)
 *
 *----------------------------------------------------------------------
 */
static void
LinuxDriverDequeue(VMLinux *vmLinux)
{
     
   HostIF_GlobalVMLock(13);	
   if (linuxState.head == vmLinux) { 
      linuxState.head = linuxState.head->next;
   } else if (linuxState.head==NULL) { 
      printk("/dev/vmmon: close -  empty VM list \n");
   } else {
      VMLinux *tmp = linuxState.head;
      while (tmp && tmp->next) { 
         if (tmp->next == vmLinux) { 
            tmp->next = tmp->next->next;
            tmp = NULL;
         } else {
            tmp = tmp->next;
         }
      }
      if (tmp) {
         printk("/dev/vmmon: close - could not find VM in list\n");
      }
   }      
   vmLinux->next = NULL;
   HostIF_GlobalVMUnLock(13);	
}






/*
 *----------------------------------------------------------------------
 *
 * CheckPadding --
 *
 *      check for expected padding --
 *      this check currently fails on the egcs compiler
 *
 * Results:
 *      
 *      TRUE if the check succeeds -- module will be loaded
 *      
 *      
 *
 * Side effects:
 *      output to kernel log on error
 *
 *----------------------------------------------------------------------
 */
static Bool
LinuxDriverCheckPadding(void)
{
   DTRWords dtr;
   uint16 *x;
   
   memset(&dtr,0,sizeof(DTR));
   dtr.dtr.limit = 0x1111;
   dtr.dtr.offset = 0x22223333;
   
   x = (uint16*)&dtr;
   
   if (x[0] == 0x1111 && x[1] == 0x3333 && x[2] == 0x2222) { 
   } else {
      Warning("DTR padding\n");
      goto error;
   }
   
   if (sizeof(VMCrossPage) != MODULECALL_CROSSPAGE_SIZE ||
       offsetof(VMCrossPage,crosspageLinearAddr)!=MODULECALL_CROSSPAGE_LAOFFSET) {
      Warning("cross page 0x%x expected 0x%x. la 0x%x expected 0x%x\n",
              sizeof(VMCrossPage), MODULECALL_CROSSPAGE_SIZE,
              offsetof(VMCrossPage,crosspageLinearAddr), MODULECALL_CROSSPAGE_LAOFFSET);
      goto error;
   }
   
   return TRUE;
   

 error:
   
   printk("/dev/vmmon: Cannot load module. Use standard gcc compiler\n");
   return FALSE;
}


#ifdef DO_APM
static int LinuxDriverAPMCallback(apm_event_t event)
{
   switch (event) {
   case APM_SYS_SUSPEND:
   case APM_USER_SUSPEND:
      if (LinuxDriverAPMstate == APM_STATE_READY) {
	 // call up to user to suspend VMs
      }
      break;
   case APM_NORMAL_RESUME:
   case APM_CRITICAL_RESUME:
      if (LinuxDriverAPMstate == APM_STATE_SUSPEND) {
	 // call up to user to resume VMs
      }
   }
   return 0;
}
#endif

