/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: x86.h,v 1.6 2000/01/15 01:02:19 chuckr Exp $
 * **********************************************************/

#ifndef _X86_H_
#define _X86_H_

#include "vm_types.h"


#define SIZE_8BIT  1
#define SIZE_16BIT 2
#define SIZE_24BIT 3
#define SIZE_32BIT 4
#define SIZE_48BIT 6
#define SIZE_64BIT 8
#define SIZE_80BIT 10
#define SIZE_128BIT 16



/*
 * simple structures are defined but shift and masks where more
 * complicated structures use structs and bitmasks
 */


/*
 *   control registers
 */

#define CR0_PE         0x00000001
#define CR0_MP         0x00000002
#define CR0_EM         0x00000004
#define CR0_TS         0x00000008
#define CR0_ET         0x00000010
#define CR0_NE         0x00000020
#define CR0_WP         0x00010000
#define CR0_AM         0x00040000
#define CR0_NW         0x20000000
#define CR0_CD         0x40000000
#define CR0_PG         0x80000000
#define CR0_RESERVED   0x1ffaffc0
#define CR0_NOT_IMP    0x00000000

#define CR3_PWT        0x00000008
#define CR3_PCD        0x00000010
#define CR3_PDB_SHIFT  12
#define CR3_PDB_MASK   0xfffff000
#define CR3_RESERVED   0x00000fe7
#define CR3_NOT_IMP    0x00000018

#define CR4_VME        0x00000001
#define CR4_PVI        0x00000002
#define CR4_TSD        0x00000004
#define CR4_DE         0x00000008
#define CR4_PSE        0x00000010
#define CR4_PAE        0x00000020
#define CR4_MCE        0x00000040
#define CR4_PGE        0x00000080
#define CR4_PCE        0x00000100

#define CR4_OSFXSR     0x00000200 /* CPU/OS supports SIMD insts */
#define CR4_OSXMMEXCPT 0x00000400 /* #XF exception enable PIII only */

#define CR4_RESERVED   0xfffff800
#define CR4_NOT_IMP    0x00000026

#define CR4_RESERVED_WO_KNI   (CR4_RESERVED|CR4_OSXMMEXCPT)
#define CR4_NOT_IMP_WO_KNI    (CR4_NOT_IMP|CR4_OSXMMEXCPT)

#define EFLAGS_CF      0x00000001
#define EFLAGS_SET     0x00000002
#define EFLAGS_PF      0x00000004
#define EFLAGS_AF      0x00000010
#define EFLAGS_ZF      0x00000040
#define EFLAGS_SF      0x00000080
#define EFLAGS_TF      0x00000100
#define EFLAGS_IF      0x00000200
#define EFLAGS_DF      0x00000400
#define EFLAGS_OF      0x00000800
#define EFLAGS_IOPL    0x00003000
#define EFLAGS_NT      0x00004000
#define EFLAGS_RF      0x00010000
#define EFLAGS_VM      0x00020000
#define EFLAGS_AC      0x00040000
#define EFLAGS_VIF     0x00080000
#define EFLAGS_VIP     0x00100000
#define EFLAGS_ID      0x00200000

#define EFLAGS_ALL       (0x003f7fd5)

#define EFLAGS_REAL_32   (EFLAGS_ALL & ~(EFLAGS_VIP|EFLAGS_VIF|EFLAGS_VM))
#define EFLAGS_V8086_32  (EFLAGS_ALL & ~(EFLAGS_VIP|EFLAGS_VIF|EFLAGS_VM|EFLAGS_IOPL))
#define EFLAGS_ALL_16    LOWORD(EFLAGS_ALL)
#define EFLAGS_REAL_16   LOWORD(EFLAGS_REAL_32)
#define EFLAGS_V8086_16  LOWORD(EFLAGS_V8086_32)
#define EFLAGS_CC        0x00000cd5

#define EFLAGS_CLEAR_ON_EXC     (EFLAGS_TF|EFLAGS_VM|EFLAGS_RF|EFLAGS_NT)

#define EFLAGS_IOPL_SHIFT 12
#define EFLAGS_IOPL_MASK  0x3000


#define DR6_B0         0x00000001
#define DR6_B1         0x00000002
#define DR6_B2         0x00000004
#define DR6_B3         0x00000008
#define DR6_BD         0x00002000
#define DR6_BS         0x00004000
#define DR6_BT         0x00008000
#define DR6_ONES       0xffff0ff0
#define DR6_RESERVED_MASK 0xffff1ff0

#define DR7_L0         0x00000001
#define DR7_G0         0x00000002
#define DR7_L1         0x00000004
#define DR7_G1         0x00000008
#define DR7_L2         0x00000010
#define DR7_G2         0x00000020
#define DR7_L3         0x00000040
#define DR7_G3         0x00000080
#define DR7_LE         0x00000100
#define DR7_GE         0x00000200
#define DR7_GD         0x00002000

#define DR7_RW(_r,_n)  (((_r) >> (16+(_n)*4)) & 0x3)
#define DR7_L(_r,_n)   (((_r) >> ((_n)*2)) & 1)
#define DR7_G(_r,_n)   (((_r) >> (1 + (_n)*2)) & 1)
#define DR7_LEN(_r,_n) (((_r) >> (18+(_n)*4)) & 0x3)

#define DR7_RW_INST    0x0
#define DR7_RW_WRITES  0x1
#define DR7_RW_IO      0x2
#define DR7_RW_ACCESS  0x3

#define DR7_ONES       0x00000400



/* www.sandpile.org/80x86/cpuid.shtml */
#define CPUID_FEATURE_FPU    0x00000001
#define CPUID_FEATURE_VME    0x00000002
#define CPUID_FEATURE_DBGE   0x00000004
#define CPUID_FEATURE_PGSZE  0X00000008
#define CPUID_FEATURE_TSC    0x00000010
#define CPUID_FEATURE_MSR    0x00000020
#define CPUID_FEATURE_PAE    0x00000040
#define CPUID_FEATURE_MCK    0x00000080
#define CPUID_FEATURE_CPMX   0x00000100
#define CPUID_FEATURE_APIC   0x00000200
#define CPUID_FEATURE_RSVD1  0x00000400
#define CPUID_FEATURE_SEP    0x00000800
#define CPUID_FEATURE_MTTR   0x00001000
#define CPUID_FEATURE_PGE    0x00002000 
#define CPUID_FEATURE_MCA    0x00004000
#define CPUID_FEATURE_CMOV   0x00008000 
#define CPUID_FEATURE_PAT    0x00010000
#define CPUID_FEATURE_36PG   0x00020000
#define CPUID_FEATURE_RSVD2  0x00040000 /* pentium II, mendocino? */
#define CPUID_FEATURE_RSVD3  0x00080000
#define CPUID_FEATURE_RSVD4  0x00100000
#define CPUID_FEATURE_RSVD5  0x00200000
#define CPUID_FEATURE_RSVD6  0x00400000
#define CPUID_FEATURE_MMX    0x00800000 
#define CPUID_FEATURE_FXSAVE 0x01000000
#define CPUID_FEATURE_XMM    0x02000000  /* pentium III */
#define CPUID_FEATURE_RSVD   0xfc780000 

#define CPUID_FAMILY(_eax)   (((_eax) >> 8) & 0xf)

#define CPUID_FAMILY_486     4
#define CPUID_FAMILY_P5      5
#define CPUID_FAMILY_P6      6

#ifndef MSR_TSC
#define MSR_TSC              0x00000010
#endif
#define MSR_MTRR_CAP         0x000000fe
#if __NetBSD_Version__ <= 106120000
#define MSR_SYSENTER_CS      0x00000174
#define MSR_SYSENTER_ESP     0x00000175
#define MSR_SYSENTER_EIP     0x00000176
#endif


typedef uint32 CReg;


/*
 *   segment selectors
 */

typedef uint16 Selector;

#define SELECTOR_GDT             0
#define SELECTOR_LDT             1
#define SELECTOR_RPL_SHIFT       0
#define SELECTOR_RPL_MASK        0x03
#define SELECTOR_TI_SHIFT        2
#define SELECTOR_TI_MASK         0x4
#define SELECTOR_INDEX_SHIFT     3
#define SELECTOR_INDEX_MASK      0xfff8

#define SELECTOR_RPL(_sel)       ((_sel) & SELECTOR_RPL_MASK)
#define SELECTOR_TABLE(_sel)     (((_sel) & SELECTOR_TI_MASK) >> SELECTOR_TI_SHIFT)
#define SELECTOR_INDEX(_sel)     ((_sel) >> SELECTOR_INDEX_SHIFT)

#define NULL_SELECTOR(_sel)      (!((_sel) & ~SELECTOR_RPL_MASK))

/*
 *   tasks
 */

#ifdef __GNUC__
struct Task {
#else
typedef struct Task {
#endif
   uint16     prevTask,  __prevTasku;
   uint32     esp0;
   uint16     ss0,  __ss0u;
   uint32     esp1;
   uint16     ss1,  __ss1u;
   uint32     esp2;
   uint16     ss2,  __ss2u;
   uint32     cr3;
   uint32     eip;
   uint32     eflags;
   uint32     eax;
   uint32     ecx;
   uint32     edx;
   uint32     ebx;
   uint32     esp;
   uint32     ebp;
   uint32     esi;
   uint32     edi;
   uint16     es,  __esu;
   uint16     cs,  __csu;
   uint16     ss,  __ssu;
   uint16     ds,  __dsu;
   uint16     fs,  __fsu;
   uint16     gs,  __gsu;
   uint16     ldt,  __ldtu;
   uint16     trap;
   uint16     IOMapBase;
#ifdef __GNUC__
}  __attribute__ ((packed));

typedef struct Task Task;
#else
} Task;
#endif


typedef struct {
   uint16     prevTask;
   uint16     sp0;
   uint16     ss0;
   uint16     sp1;
   uint16     ss1;
   uint16     sp2;
   uint16     ss2;
   uint16     ip;
   uint16     flags;
   uint16     ax;
   uint16     cx;
   uint16     dx;
   uint16     bx;
   uint16     sp;
   uint16     bp;
   uint16     si;
   uint16     di;
   uint16     es;
   uint16     cs;
   uint16     ss;
   uint16     ds;
   uint16     fs;
   uint16     gs;
   uint16     ldt;
} Task16;


/*
 *   segment descriptors
 */

typedef struct Descriptor {
   unsigned   limit_lo  : 16;
   unsigned   base_lo   : 16;
   unsigned   base_mid  : 8;
   unsigned   type      : 4;
   unsigned   S         : 1;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   limit_hi  : 4;
   unsigned   AVL       : 1;
   unsigned   unused    : 1;
   unsigned   DB        : 1;
   unsigned   gran      : 1;
   unsigned   base_hi   : 8;
} Descriptor;

typedef union {
   Descriptor desc;
   uint32     word[2];
} DescriptorUnion;


#define EXPANDEDLIMIT(_desc)((_desc).gran? (LIMIT(_desc) << 12) | 0xfff : LIMIT(_desc))
#define LIMIT(_desc)        (((_desc).limit_hi<<16) | (_desc).limit_lo)
#define BASE(_desc)         (((_desc).base_hi<<24) | ((_desc).base_mid<<16) | (_desc).base_lo)

#define LIMIT_LO(_dw)       (((int)(_dw)) & 0xffff)
#define LIMIT_HI(_dw)       ((((int)(_dw)) >> 16) & 0xf)

#define BASE_LO(_dw)        (((int)(_dw)) & 0xffff)
#define BASE_MID(_dw)       ((((int)(_dw)) >> 16) & 0xff)
#define BASE_HI(_dw)        ((((int)(_dw)) >> 24) & 0xff)

#define DT_CODE(_desc)               ( ((_desc).S) && (((_desc).type & 0x8) == 0x8))
#define DT_CONFORMING_CODE(_desc)    ( ((_desc).S) && (((_desc).type & 0xc) == 0xc))
#define DT_NONCONFORMING_CODE(_desc) ( ((_desc).S) && (((_desc).type & 0xc) == 0x8))
#define DT_READABLE_CODE(_desc)      ( ((_desc).S) && (((_desc).type & 0xa) == 0xa))
#define DT_DATA(_desc)               ( ((_desc).S) && (((_desc).type & 0x8) == 0x0))
#define DT_WRITEABLE_DATA(_desc)     ( ((_desc).S) && (((_desc).type & 0xa) == 0x2))
#define DT_EXPAND_DOWN(_desc)        ( ((_desc).S) && (((_desc).type & 0xc) == 0x4))
#define DT_CALL_GATE(_desc)          (!((_desc).S) && (((_desc).type & 0x7) == 0x4))
#define DT_LDT(_desc)                (!((_desc).S) && (((_desc).type & 0xf) == 0x2))
#define DT_TASK_GATE(_desc)          (!((_desc).S) && (((_desc).type & 0xf) == 0x5))
#define DT_TSS(_desc)                (!((_desc).S) && (((_desc).type & 0x5) == 0x1))
#define DT_AVAIL_TSS(_desc)          (!((_desc).S) && (((_desc).type & 0x7) == 0x1))

#define DT_ACCESS                    0x2
#define DT_32BIT                     0x8
#define DT_TSS_BUSY                  0x2

#define GT_TASK(_gate)               (((_gate).type & 0x1f) == 0x05)
#define GT_INTR(_gate)               (((_gate).type & 0x17) == 0x06)
#define GT_TRAP(_gate)               (((_gate).type & 0x17) == 0x07)
#define GT_32BIT                     0x08

#define DATA_DESC           0x2
#define CODE_DESC           0xa
#define TASK_DESC           0x9  /* TTS available */
#define TASK_DESC_BUSY      0xb  /* TSS busy */


/*
 *----------------------------------------------------------------------
 *
 * Desc_Limit --
 *
 *      Return the limit in bytes of the descriptor.
 *
 * Results:
 *      The limit in bytes of the descriptor.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE uint32
Desc_Limit(Descriptor *descp)
{
   return (descp->gran) ? (LIMIT(*descp) << 12) | 0xfff : LIMIT(*descp);
}


/*
 *----------------------------------------------------------------------
 *
 * Desc_InBounds and Desc_LegalAccess --
 *
 *      Check whether the access is within the bounds of the segment.
 *
 * Results:
 *      TRUE is the access is within the bouns of the segment, FALSE
 *      otherwise.
 *
 * Side effects:
 *      None.
 *
 * Bugs:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Bool
Desc_InBounds(VA vaddr, VA limit, uint32 size, Bool expandDown, unsigned DB) {
   if (expandDown) {
      VA end = DB ? 0xffffffff : 0xffff; 
      return  ((vaddr >= limit) && (vaddr <= end) && ((size-1) <= (end - vaddr)));
   } else {
      return  ((vaddr <= limit) && ((size-1) <= (limit - vaddr)));
   }
}

static INLINE Bool
Desc_LegalAccess(Descriptor *descp, VA vaddr, uint32 size)
{
   return Desc_InBounds(vaddr,Desc_Limit(descp),
                        size,DT_EXPAND_DOWN(*descp),descp->DB);
}


/*
 *   call gates
 */

typedef struct {
   unsigned   offset_lo : 16;
   unsigned   segment   : 16;
   unsigned   params    : 5;
   unsigned   unused    : 3;
   unsigned   type      : 5;
   unsigned   DPL       : 2;
   unsigned   present   : 1;
   unsigned   offset_hi : 16;
} Gate;

#define GATE_OFFSET(_gate)       (((_gate).offset_hi << 16) | (_gate).offset_lo)

#define GATE_OFFSET_LO(_dw)      (((int)(_dw)) & 0xffff)
#define GATE_OFFSET_HI(_dw)      ((((int)(_dw)) >> 16) & 0xffff)

#define TASK_GATE           0x05
#define INTER_GATE          0x0e
#define TRAP_GATE           0x0f


/*
 *   descriptor table registers
 */

/* Need to pack the DTR struct so the
   offset starts right after the limit */

#ifdef __GNUC__

typedef struct DTR {
   uint16 limit;
   uint32 offset __attribute__ ((packed));
} DTR;

#elif _MSC_VER

#pragma pack(push)
#pragma pack(2)

typedef struct {
   uint16 limit;
   uint32 offset;
} DTR;

#pragma pack(pop)

#else
# error
#endif

typedef union {
   DTR dtr;
   uint32 word[2];
} DTRWords;

typedef struct {
   uint32 va;
   Selector seg;
} FarPtr;


/*
 *   far pointers
 */

typedef struct FarPtr16 {
   uint16   offset;
   uint16   selector;
} FarPtr16;

typedef struct FarPtr32 {
   uint32   offset;
   uint16   selector;
} FarPtr32;




/*
 *   register numbering
 */

#define REG_EAX         0x00
#define REG_ECX         0x01
#define REG_EDX         0x02
#define REG_EBX         0x03
#define REG_ESP         0x04
#define REG_EBP         0x05
#define REG_ESI         0x06
#define REG_EDI         0x07


#define SEG_ES       0
#define SEG_CS       1
#define SEG_SS       2
#define SEG_DS       3
#define SEG_FS       4
#define SEG_GS       5
#define SEG_LDTR     6
#define SEG_TR       7

#define NUM_SEGS     8
   
/*
 *   exception frame.
 */


typedef struct ExcFrame {
   /* pushed by sw */
   uint16      es, __esu;
   uint16      csCopy, __csCopyu; /* Copy from HW pushed value - not restored */
   uint16      ssCopy, __ssCopyu; /* Copy from HW pushed value - not restored */
   uint16      ds, __dsu;
   uint16      fs, __fsu;
   uint16      gs, __gsu;

   Reg         eax;
   Reg         ecx;
   Reg         edx;
   Reg         ebx;
   Reg         espCopy; /* Copy from HW pushed - not restored */
   Reg         ebp;
   Reg         esi;
   Reg         edi;

   /* pushed by SW or HW depending on the interrupt */
   Reg         errorCode;
   
   /* pushed by HW */
   Reg         eip;
   uint16      cs, __csu;
   Reg         eflags;
   
   /* pushed by HW on protection level changes */
   Reg         esp;
   uint16      ss, __ssu;

   /* pushed by HW coming from virtual 8086 mode */
   uint16      v8086_es, __v8086_esu;
   uint16      v8086_ds, __v8086_dsu;
   uint16      v8086_fs, __v8086_fsu;
   uint16      v8086_gs, __v8086_gsu;
   
} ExcFrame;

/*
 *   page tables
 */

#define PTE_P           0x00000001
#define PTE_RW          0x00000002
#define PTE_US          0x00000004
#define PTE_PWT         0x00000008
#define PTE_PCD         0x00000010
#define PTE_A           0x00000020
#define PTE_D           0x00000040
#define PTE_PS          0x00000080
#define PTE_G           0x00000100

#define PTE_FLAGS       (PTE_P|PTE_RW|PTE_US)

#define PTE_PAGE_TABLE  (PTE_P|PTE_RW|PTE_US|PTE_A|PTE_D) 

#define PTE_PRESENT(_pte)   (((_pte) & PTE_P) != 0)
#define PTE_WRITEABLE(_pte) (((_pte) & PTE_RW) != 0)
#define PTE_DIRTY(_pte)     (((_pte) & PTE_D) != 0)
#define PTE_USER(_pte)      (((_pte) & PTE_US) != 0)

#define PTE_AVAIL_MASK  0xe00
#define PTE_AVAIL_SHIFT 9
#define PTE_PFN_MASK    0xfffff000
#define PTE_PFN_SHIFT   12
#define PDE_PFN_MASK    0xffc00000

#define PTE_2_PFN(_pte) (((_pte) & PTE_PFN_MASK) >> PTE_PFN_SHIFT)
#define PDE_2_PFN(_pte) (((_pte) & PDE_PFN_MASK) >> PTE_PFN_SHIFT)


#define PDES_PER_PMAP      (PAGE_SIZE/sizeof(PDE))
#define PTES_PER_PDIR      (PAGE_SIZE/sizeof(PTE))

#define PFN_2_PDOFF(_a)   ((_a) >> 10)
#define PFN_2_PGOFF(_a)   ((_a) & 0x3ff)

#define PA_2_PDOFF(_a)   ((_a) >> (10+PAGE_SHIFT))
#define PA_2_PGOFF(_a)   ((PA_2_PFN(_a)) & 0x3ff)


#define VA_2_PDOFF(_a)    ((_a) >> 22)
#define VA_2_PGOFF(_a)    (PFN_2_PGOFF(VA_2_VPN(_a)))

#define PTE_2_PA(_a)      (PPN_2_PA(PTE_2_PFN(_a)))
#define PDE_2_PA(_a)      (PPN_2_PA(PDE_2_PFN(_a)))


/*
 * Location of the APICs' registers.
 */

#define APIC_DEFAULT_ADDRESS      0xfee00000
#define IOAPIC_DEFAULT_ADDRESS    0xfec00000

#define IOAPIC_NUM_REDIR_REGS     24

#define APICR_ID         0x02
#define APICR_VERSION    0x03
#define APICR_TPR        0x08
#define APICR_APR        0x09
#define APICR_PPR        0x0a
#define APICR_EOI        0x0b
#define APICR_LDR        0x0d
#define APICR_DFR        0x0e
#define APICR_SVR        0x0f
#define APICR_ISR        0x10
#define APICR_TMR        0x18
#define APICR_IRR        0x20
#define APICR_ESR        0x28
#define APICR_ICRLO      0x30
#define APICR_ICRHI      0x31
#define APICR_TIMERLVT   0x32
#define APICR_PCLVT      0x34
#define APICR_LVT0       0x35
#define APICR_LVT1       0x36
#define APICR_ERRLVT     0x37
#define APICR_INITCNT    0x38
#define APICR_CURCNT     0x39
#define APICR_DIVIDER    0x3e

#define APICR_SIZE       0x400

#define APIC_PR_MASK          0x000000ff
#define APIC_PR_XMASK         0x000000f0
#define APIC_PR_YMASK         0x0000000f
#define APIC_PR_X(_r)         (((_r) & APIC_PR_XMASK) >> 4)
#define APIC_PR_Y(_r)         (((_r) & APIC_PR_YMASK) >> 0)

#define APIC_SVR_ONES         0x0000000f
#define APIC_SVR_VECTOR       0x000000f0
#define APIC_SVR_APICENABLE   0x00000100
#define APIC_SVR_FOCUSCHECK   0x00000200

#define APIC_BASE_MSR         0x1b

#define APIC_MSR_BASEMASK     0xfffff000
#define APIC_MSR_ENABLED      0x00000800
#define APIC_MSR_BSP          0x00000100

#define APIC_VTE_VECTOR_MASK  0x000000ff
#define APIC_VTE_MODE_FIXED   0x00000000
#define APIC_VTE_MODE_NMI     0x00000400
#define APIC_VTE_MODE_EXTINT  0x00000700
#define APIC_VTE_MODE_MASK    0x00000700
#define APIC_VTE_DELVSTATUS   0x00001000
#define APIC_VTE_PINPOL       0x00002000
#define APIC_VTE_REMIRR       0x00004000
#define APIC_VTE_TRIGMODE     0x00008000
#define APIC_VTE_MASK         0x00010000
#define APIC_VTE_TIMERMODE    0x00020000

#define APIC_ID_MASK          0x0f000000
#define APIC_ID_SHIFT         24

#define IOAPICID              0x00
#define IOAPICVER             0x01
#define IOAPICARB             0x02
#define IOREDTBL_FIRST        0x10
#define IOREDTBL_LAST         0x3f

/*
 *   exception error codes
 */

#define EXC_NONE         -1
#define EXC_BUS_ERROR    -2   /* XXX ? */

#define EXC_DE            0
#define EXC_DB            1
#define EXC_NMI           2
#define EXC_BP            3
#define EXC_OF            4
#define EXC_BR            5
#define EXC_UD            6
#define EXC_NM            7
#define EXC_DF            8
#define EXC_TS           10
#define EXC_NP           11
#define EXC_SS           12
#define EXC_GP           13
#define EXC_PF           14
#define EXC_MF           16
#define EXC_AC           17
#define EXC_MC           18
#define EXC_XF           19  // SIMD exception

#define PF_P            0x1
#define PF_RW           0x2
#define PF_US           0x4
#define PF_RSVD         0x8


/*
 *  instruction decoding
 */

#define MODRM_MOD(_m)  (((_m) >> 6) & 0x3)
#define MODRM_REG(_m)  (((_m) >> 3) & 0x7)
#define MODRM_RM(_m)   (((_m) >> 0) & 0x7)

/*
 * Debug registers
 */

#define DEBUG_STATUS_B0   (1<<0)
#define DEBUG_STATUS_B1   (1<<1)
#define DEBUG_STATUS_B2   (1<<2)
#define DEBUG_STATUS_B3   (1<<3)
#define DEBUG_STATUS_DB   (1<<13)
#define DEBUG_STATUS_BS   (1<<14)
#define DEBUG_STATUS_BT   (1<<15)

typedef struct DebugControlRegister {

   int l0:1;
   int g0:1;
   int l1:1;
   int g1:1;
   int l2:1;
   int g2:1;
   int l3:1;
   int g3:1;
   
   int le:1;
   int ge:1;
   int oo1:3;
   
   int gd:1;
   int oo:2;
   
   int rw0:2;
   int len0:2;
   int rw1:2;
   int len1:2;
   int rw2:2;
   int len2:2;
   int rw3:2;
   int len3:2;
   
} DebugControlRegister;

  

                 
#define MAKE_SELECTOR(_index, _ti, _RPL)   \
( (((_index) << SELECTOR_INDEX_SHIFT) & SELECTOR_INDEX_MASK) \
  | (((_ti)    << SELECTOR_TI_SHIFT)    & SELECTOR_TI_MASK) \
  | (((_RPL)   << SELECTOR_RPL_SHIFT)   & SELECTOR_RPL_MASK) )


#define MAKE_PTE(_mpfn, _avail, _flags) \
                        (((_mpfn) << PTE_PFN_SHIFT) \
                         | (((_avail) & PTE_AVAIL_MASK) << PTE_AVAIL_SHIFT) \
                         | (_flags))


/*
 * SMM State Save Map 
 * See the Intel processor manual for documentation.
 * The extract layout of the structure is guesswork.  It should really
 * matter also as the Intel documented fields are at the right offset. 
 */

/*
 * SMMSegState - Store the "hidden" state of a segment descriptor in 
 * the SMM save state map. NOTE: Format is somewhat defined by the 
 * Intel hardware.
 */
typedef struct SMMSegState {
  uint32 limit;   /* Limit from descriptor */
  uint32 base;    /* Base from descriptor */
  uint32 rights;  /* Rest of the descriptor */
} SMMSegState;

/*
 * Format the the SMM save state map.
 * NOTE: This too is defined by the Intel hardware.
 */

typedef struct SMMStateSaveMap {
  uint8  reserved0[248];  
  uint32 smbase;
  uint32 revId; 
  uint16 ioInstRestart;
  uint16 haltRestart;
  uint8  reserved01[12];
  uint32 ioInstEIP; 
  uint8  reserved1[18];
  uint32 cr4reg;
  uint8  reserved2[4];
  SMMSegState hiddenSegPart[7];
  SMMSegState gdtr;
  SMMSegState idtr;
  SMMSegState hiddenTRpart;
  uint32 segRegs[8];
  Reg    dr7reg;
  Reg    dr6reg;
  Reg    gpRegs[8];
  Reg    eip;
  Reg    eflags;
  Reg    cr3reg;
  Reg    cr0reg;
} SMMStateSaveMap;

#define SMM_SAVEMAP_SIZE   512
#define SMRAM_DEFAULT_BASE 0x30000
#define SMRAM_ENTRY_POINT 0x8000
#define SMRAM_SAVE_STATE_OFFSET 0xfe00

#define SMM_RIGHTS_BITS   0x00f0ff00

#endif _X86_H_

