/*
 * Copyright 2001 Wasabi Systems, Inc.
 * All rights reserved.
 *
 * Written by Frank van der Linden for Wasabi Systems, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed for the NetBSD Project by
 *      Wasabi Systems, Inc.
 * 4. The name of Wasabi Systems, Inc. may not be used to endorse
 *    or promote products derived from this software without specific prior
 *    written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY WASABI SYSTEMS, INC. ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL WASABI SYSTEMS, INC
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * Inspiration from code by Vladimir N. Silyaev who wrote something like
 * this for FreeBSD.
 */


/*
 * Awful code to try to emulate a small part of the linux rtc device.
 * Under Linux, this device allows a user program to set the timer to
 * a higher frequency, and get interrupts from it.
 *
 * This is 'simulated' here with a busy wait (ugh!), but I couldn't
 * see any other way to get sub-tick resolution.
 *
 * Fortunately, this isn't used often by VMware. Also, a check was built
 * in to the 'set' operation to deny attempts at setting a higher interval
 * than our timeslice (otherwise this device could be a nice DoS).
 */

#include <sys/param.h>
#include <sys/conf.h>
#include <sys/device.h>
#include <sys/filedesc.h>
#include <sys/file.h>
#include <sys/filio.h>
#include <sys/select.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/poll.h>
#include <sys/proc.h>
#include <sys/systm.h>
#include <sys/ioctl.h>
#include <sys/exec.h>
#include <sys/lkm.h>

#include "rtcvar.h"

int rtc_lkmentry(struct lkm_table *, int, int);
static int rtc_handle(struct lkm_table *, int);

static int rtc_open(dev_t dev, int oflags, int devtype, struct proc *p);
static int rtc_close(dev_t dev, int cflags, int devtype, struct proc *p);
static int rtc_ioctl(dev_t dev, u_long cmd, caddr_t data, int flags,
		 struct proc *p);
static int rtc_poll(dev_t, int, struct proc *);

struct rtc_softc rtc_sc;

static struct cdevsw rtc_dev = {
	rtc_open, rtc_close, 
	(dev_type_read((*)))enodev, (dev_type_write((*)))enodev,
	rtc_ioctl, (dev_type_stop((*))) enodev, 0,
	rtc_poll, (dev_type_mmap((*))) enodev, 0
};


#if __NetBSD_Version__ >= 106080000
MOD_DEV("linuxrtc", "linuxrtc", NULL, -1, &rtc_dev, -1)
#else
MOD_DEV("linuxrtc", LM_DT_CHAR, -1, &rtc_dev)
#endif


int
rtc_lkmentry(struct lkm_table *lkmtp, int cmd, int ver)
{
	DISPATCH(lkmtp, cmd, ver, rtc_handle, rtc_handle, rtc_handle)
}

static int 
rtc_handle(struct lkm_table *lkmtp, int cmd)
{
	int error = 0;

	switch (cmd) {
	case LKM_E_LOAD:
		if (lkmexists(lkmtp)) 
			return EEXIST;
		break;
		
	case LKM_E_UNLOAD:
		if (rtc_sc.sc_flags & RTC_OPEN)
			return (EBUSY);
		break;
		
	case LKM_E_STAT:
		break;

	default:
		error = EIO;
		break;
	}
	return error;
}

static int
rtc_open(dev_t dev, int flag, int mode, struct proc *p)
{
	if (rtc_sc.sc_flags & RTC_OPEN)
		return EBUSY;

	if (suser(p->p_ucred, &p->p_acflag) != 0)
		return EPERM;

	rtc_sc.sc_flags = RTC_OPEN;
	rtc_sc.sc_freq = 1000000 / hz;

	return 0;
}


static int
rtc_close(dev_t dev, int flags, int mode, struct proc *p)
{
	rtc_sc.sc_flags = 0;

	return 0;
}


/*
 * This function uses the hack to make PTIOCLINUX ioctls (passthroughs
 * from the emulation) return EJUSTRETURN to be able to have them
 * set syscall return values for the benefit of Linux emulation.
 */
static int
rtc_ioctl(dev_t dev, u_long cmd, caddr_t data, int flag, struct proc *p)
{
	struct ioctl_pt *pt;
	int val;

	if (cmd != PTIOCLINUX)
		return ENOTTY;

	pt = (struct ioctl_pt *)data;

	switch (pt->com) {
	case LINUX_RTC_PIE_ON:
		rtc_sc.sc_flags |= RTC_ACTIVE;
		break;
	case LINUX_RTC_PIE_OFF:
		rtc_sc.sc_flags &= ~RTC_ACTIVE;
		break;
	case LINUX_RTC_IRQP_SET:
		val = (int)pt->data;
		if (val <= 1 || val >= 8192)
			return EINVAL;
		rtc_sc.sc_freq = (int)val;
		break;
	default:
		return ENOTTY;
	}
	return (0);
}

static int
rtc_poll(dev_t dev, int events, struct proc *p)
{
	int del, revents;

	if (!(rtc_sc.sc_flags & RTC_ACTIVE))
		return 0;

	del = revents = 0;

	/*
	 * Yes, this is awful. I don't see another reasonable
	 * way to do this.
	 */
	if (events != 0) {
		del = 1000000 / rtc_sc.sc_freq;
		DELAY(del);
		revents = events;
	}
	return revents;
}
